package main.service;

import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import main.exception.IncidenceNotFoundException;
import main.gestiona.apirest.GestionaAPIService;
import main.gestiona.apirest.config.ProceduresConfig;
import main.gestiona.apirest.model.FileModel;
import main.gestiona.apirest.model.RegistryAnnotationModel;
import main.gestiona.apirest.model.ThirdPartyModel;
import main.model.IncidenceModel;
import main.model.IncidenceModel.IncidentStatus;
import main.repository.IncidenceRepository;

@Service
public class IncidenceService {

	private final IncidenceRepository repository;
	private final String procedureName;
	private final String externalProcedureName;

	@Autowired
	private GestionaAPIService gestionaAPIService;

	@Autowired
	public IncidenceService(IncidenceRepository repository, ProceduresConfig proceduresConfig) {
		this.repository = repository;
		this.procedureName = proceduresConfig.getName();
		this.externalProcedureName = proceduresConfig.getExternalProcedure();

	}

	/**
	 * Obtiene el listado de incidencias de la base de datos
	 *
	 * @return el listado con los modelos obtenidos
	 */
	public List<IncidenceModel> getAll() {
		return repository.findAll();
	}

	/**
	 * Persiste una nueva incidencia en la base de datos
	 *
	 * @param incidenceModel
	 * 		el modelo a guardar
	 */
	public void create(IncidenceModel incidenceModel) {
		repository.save(incidenceModel);
	}

	/**
	 * Modifica la incidencia en la base de datos en caso de que exista, en caso contrario lanza una excepción
	 *
	 * @param incidenceModel
	 * 		el modelo a actualizar
	 *
	 * @throws IncidenceNotFoundException
	 * 		en el caso de que no encuentre la incidencia enviada por parámetro
	 */
	public void update(IncidenceModel incidenceModel) throws IncidenceNotFoundException {

		if (!repository.existsById(incidenceModel.getId())) {
			throw new IncidenceNotFoundException(String.format("No se ha encontrado la incidencia con id %s", incidenceModel.getId()));
		}

		repository.save(incidenceModel);
	}

	/**
	 * Busca una incidencia por número de expediente
	 *
	 * @param fileNumber
	 * 		número de expediente
	 *
	 * @return el modelo de incidencia encontrado, o lanza una excepción en caso de que no exista
	 *
	 * @throws IncidenceNotFoundException
	 * 		en caso de que no encuentre la incidencia
	 */
	public IncidenceModel searchIncidence(String fileNumber) throws IncidenceNotFoundException {

		return repository.findByGestionaFileNumber(fileNumber)//
				.orElseThrow(() -> new IncidenceNotFoundException(String.format("No se ha encontrado la incidencia con número de expediente %s", fileNumber)));
	}

	/**
	 * Busca una incidencia en la base de datos a través de su número de incidencia
	 *
	 * @param incidenceNumber
	 * 		número de la incidencia a buscar
	 *
	 * @return el modelo de la incidencia
	 *
	 * @throws IncidenceNotFoundException
	 * 		en caso de que no encuentre la incidencia
	 */
	public IncidenceModel getIncidenceByNumber(String incidenceNumber) throws IncidenceNotFoundException {
		return repository.findByIncidenceNumber(incidenceNumber).orElseThrow(
				() -> new IncidenceNotFoundException(String.format("No se ha encontrado la incidencia con número %s", incidenceNumber)));

	}

	/**
	 * Cambia el estado de una incidencia a ASSIGNED y le establece su prioridad y lo persiste en base de datos
	 *
	 * @param incidenceModel
	 * 		modelo de la incidencia a modificar
	 */
	public void setAssignedIncidence(IncidenceModel incidenceModel, int priority) {
		incidenceModel.setStatus(IncidentStatus.ASSIGNED);
		incidenceModel.setPriority(priority);
		repository.save(incidenceModel);
	}

	/**
	 * Cambia el estado de una incidencia a RESOLVED si así se indica y lo persiste en base de datos
	 *
	 * @param incidenceModel
	 * 		modelo de la incidencia a modificar
	 */
	public void setResolvedIncidence(IncidenceModel incidenceModel, boolean setToResolved) {
		if (setToResolved) {
			incidenceModel.setStatus(IncidentStatus.RESOLVED);
			repository.save(incidenceModel);
		}
	}

	/**
	 * Registra una nueva incidencia en Gestiona. Para ello:
	 * <ol>
	 *     <li>Crea un el tercero en Gestiona si no existía</li>
	 *     <li>Crea el registro de entrada en Gestiona</li>
	 *     <li>Le asigna el tercero al registro de entrada</li>
	 *     <li>Finaliza el registro de entrada</li>
	 *     <li>Crea un nuevo expediente con el procedimiento indicado</li>
	 *     <li>Le asigna el registro al nuevo expediente</li>
	 *     <li>Crea una nueva incidencia en base de datos con los datos obtenidos en el proceso</li>
	 * </ol>
	 *
	 * @param nif
	 * 		dni del tercero al que se le asigna la incidencia
	 * @param subject
	 * 		asunto de la incidencia
	 */
	public void registerNewIncidence(String nif, String subject) {
		// Comprueba si existe el tercero en Gestiona y si no lo crea
		gestionaAPIService.checkIfThirdExists(nif);

		// Crea el modelo del registro
		final RegistryAnnotationModel registryModel = gestionaAPIService.createInputRegistry(subject);

		// Una vez creado, saca el recurso `thirdparties` de los links del modelo y con este crea el tercero en cuestión
		final ThirdPartyModel thirdPartyModel = gestionaAPIService.getThirdPartyByNif(nif);
		gestionaAPIService.addThirdPartyToInputRegistry(registryModel, thirdPartyModel);

		// Finaliza el registro
		gestionaAPIService.finalizeInputRegistry(registryModel);

		// Refresca el modelo del registro para obtener los links necesarios para crear el expediente
		final RegistryAnnotationModel refreshedRegistryModel = gestionaAPIService.refreshRegistryAnnotationModel(registryModel);

		// Crea un nuevo expediente con el procedimiento indicado
		final FileModel fileModel = gestionaAPIService.createFileWithProcedure(procedureName, externalProcedureName);

		// Asigna el registro al expediente que hemos creado
		gestionaAPIService.addRegistryToFile(fileModel, refreshedRegistryModel);

		// Crea un nuevo modelo incidencia
		repository.save(new IncidenceModel(nif, subject, registryModel.getCode(), fileModel.getCode(), IncidentStatus.UNASSIGNED));
	}

}

