package main.service;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import main.config.ServerConfig;
import main.exception.DocumentaryFlowNotFoundException;
import main.exception.FileNotFoundException;
import main.exception.GestionaAPIException;
import main.gestiona.apirest.GestionaAPIService;
import main.gestiona.apirest.model.CircuitTemplateModel;
import main.gestiona.apirest.model.DocumentModel;
import main.gestiona.apirest.model.DocumentProcessStateModel;
import main.gestiona.apirest.model.DocumentProcessStateModel.DocumentProcessState;
import main.gestiona.apirest.model.FileDocumentAnnotationModel;
import main.gestiona.apirest.model.FileModel;
import main.gestiona.apirest.model.ThirdOutputModel;
import main.model.DocumentaryFlowModel;
import main.model.DocumentaryFlowModel.Status;
import main.repository.DocumentaryFlowRepository;

@Service
public class DocumentaryFlowService {

	private static final Logger log = LoggerFactory.getLogger(DocumentaryFlowService.class);

	private final DocumentaryFlowRepository repository;

	private final String uploadPath;

	@Autowired
	private GestionaAPIService gestionaAPIService;

	@Autowired
	public DocumentaryFlowService(DocumentaryFlowRepository repository, ServerConfig serverConfig) {
		this.repository = repository;
		this.uploadPath = serverConfig.getFileUploadPath();
	}

	/**
	 * Obtiene el listado de documentaryFlows de la base de datos
	 *
	 * @return el listado con los modelos obtenidos
	 */
	public List<DocumentaryFlowModel> getAll() {
		return repository.findAll();
	}

	/**
	 * Obtiene el flujo documental a través de su identificador
	 *
	 * @param id
	 * 		identificador del flujo documental
	 *
	 * @return el modelo del flujo documental
	 */
	public DocumentaryFlowModel getById(String id) {
		return repository.findById(id).orElse(null);
	}

	/**
	 * Sincroniza la base de datos con los expedientes obtenidos de la API de Gestiona
	 */
	public void documentaryFlowDatabaseSync() {

		final List<FileModel> fileModelList = gestionaAPIService.getFileList();

		for (FileModel fileModel : fileModelList) {

			if (!hasDocumentaryFlow(fileModel)) {
				log.info("Creando nuevo flujo documental para el expediente: {}", fileModel.getCode());
				final DocumentaryFlowModel newModel = new DocumentaryFlowModel(fileModel.getCode(), fileModel.getId());
				create(newModel);
			}
		}
	}

	/**
	 * Comprueba los estados de tramitación de los documentos asociados a los flujos documentales que se encuentren en estado `IN_PROCESS` y en el caso de que
	 * hayan terminado los pasa al estado `PROCESSED`
	 */
	public void updateProcessStatus() {
		final List<DocumentaryFlowModel> documentaryFlowList = this.getAll();

		for (DocumentaryFlowModel documentaryFlow : documentaryFlowList) {

			try {
				if (Status.IN_PROCESS.equals(documentaryFlow.getState())) {
					final String documentReference = documentaryFlow.getDocumentReference();
					final DocumentProcessStateModel documentProcessStateModel = gestionaAPIService.checkDocumentProcessStatus(documentReference);

					final DocumentProcessState documentProcessState = documentProcessStateModel.getState();
					if (!DocumentProcessState.IN_PROGRESS.equals(documentProcessState)) {
						setDocumentFlowToFinalState(documentaryFlow, documentReference, documentProcessState);
						log.info("Ha finalizado la tramitación del flujo documental con código de expediente {}", documentaryFlow.getFileCodeGestiona());
					}
				}

			} catch (GestionaAPIException e) {
				log.error("Error al obtener el estado del flujo documental para el expediente: {}", documentaryFlow.getFileIdGestiona(), e);
			}
		}
	}

	/**
	 * Obtiene el listado de circuitos de tramitación disponibles para un documento
	 *
	 * @param documentReference
	 * 		url al recurso donde se encuentra el documento en Gestiona
	 *
	 * @return el listado de circuitos de tramitación
	 */
	public List<CircuitTemplateModel> getCircuitTemplates(String documentReference) {
		return gestionaAPIService.getCircuitTemplates(documentReference);
	}

	/**
	 * Persiste un nuevo documentaryFlow en la base de datos
	 *
	 * @param documentaryFlow
	 * 		el modelo a guardar
	 */
	public void create(DocumentaryFlowModel documentaryFlow) {
		repository.save(documentaryFlow);
	}

	/**
	 * Modifica el documentaryFlow en la base de datos en caso de que exista, en caso contrario lanza una excepción
	 *
	 * @param documentaryFlow
	 * 		el modelo a actualizar
	 *
	 * @throws DocumentaryFlowNotFoundException
	 * 		en el caso de que no encuentre el flujo documental enviado por parámetro
	 */
	public void update(DocumentaryFlowModel documentaryFlow) throws DocumentaryFlowNotFoundException {

		if (!repository.existsById(documentaryFlow.getId())) {
			throw new DocumentaryFlowNotFoundException(String.format("No se ha encontrado el flujo documental con id %s", documentaryFlow.getId()));
		}

		repository.save(documentaryFlow);
	}

	/**
	 * Encargado de gestionar la subida de un documento a Gestiona, almacenarlo en el servidor y cambiar el estado del flujo documental a `UPLOADED`
	 *
	 * @param id
	 * 		identificador del flujo documental
	 * @param fileIdGestiona
	 * 		id del expediente en Gestiona
	 * @param document
	 * 		documento seleccionado para ser tramitado
	 */
	public void handleDocumentUpload(String id, String fileIdGestiona, MultipartFile document) {

		if (!document.isEmpty()) {

			try {
				// Guarda el documento en el servidor
				final String absolutePath = saveFile(document);

				// Saca el expediente al que añadirle el documento
				final FileModel fileModel = findFileModel(fileIdGestiona);

				log.info("Se va a añadir el documento {} al expediente {}", absolutePath, fileModel.getCode());

				// Añade el documento al expediente en Gestiona y obtiene su referencia
				final String documentReference = gestionaAPIService.addDocumentToFile(fileModel, absolutePath);

				// Guarda la referencia del documento en el DocumentaryFlowModel y actualiza su estado a `UPLOADED`
				setDocumentFlowToUploaded(id, documentReference);

				log.info("Se ha subido el documento \"{}\" y se ha añadido al expediente \"{}\"", absolutePath, fileModel.getCode());

			} catch (IOException e) {
				log.error("Error al tratar de subir el documento: {}", e.getMessage());
			}
		}
	}

	/**
	 * Procesa el documento con el circuito de tramitación escogido y comprueba si el tercero seleccionado se encuentra en Gestiona, en caso contrario, lo crea
	 * antes de tramitar el documento
	 *
	 * @param documentaryFlowId
	 * 		id del flujo documental
	 * @param documentReference
	 * 		modelo del documento a tramitar
	 * @param circuit
	 * 		circuito de tramitación con el que tramitarlo
	 * @param nif
	 * 		dni del tercero al que notificar si es necesario
	 *
	 * @throws GestionaAPIException
	 * 		si ha ocurrido algún error al tramitar el documento
	 */
	public void processDocument(String documentaryFlowId, String documentReference, CircuitTemplateModel circuit, String nif) {

		try {
			if (circuit.isSend_registry_output()) {

				gestionaAPIService.checkIfThirdExists(nif);

				final ThirdOutputModel third = gestionaAPIService.getThirdOutputByNif(nif);

				List<ThirdOutputModel> thirdOutputModelList = circuit.getThirds_ouput();

				if (thirdOutputModelList == null) {
					thirdOutputModelList = new ArrayList<>();
				}

				thirdOutputModelList.add(third);
				circuit.setThirds_ouput(thirdOutputModelList);
			}

			gestionaAPIService.processDocument(documentReference, circuit);
			setDocumentFlowToInProcess(documentaryFlowId);

		} catch (GestionaAPIException e) {
			log.error("Error al tramitar el documento: {}", documentReference);
		}
	}

	/**
	 * Descarga el documento tramitado dada la referencia a un documento en Gestiona
	 *
	 * @param documentReference
	 * 		referencia al documento en Gestiona
	 *
	 * @return respuesta con la descarga del documento
	 */
	public ResponseEntity<byte[]> downloadProcessedDocument(String documentReference) {

		final DocumentModel documentModel = gestionaAPIService.getDocument(documentReference);

		final byte[] documentContent = gestionaAPIService.getDocumentContent(documentModel);

		return createDocumentPetition(documentContent, documentModel.getFile_name_with_extension());
	}

	/**
	 * Cambia el estado del flujo documental con identificador "id" a "Documento subido"
	 *
	 * @param id
	 * 		identificador del modelo de flujo documental
	 *
	 * @throws DocumentaryFlowNotFoundException
	 * 		en el caso de que no se haya encontrado el flujo documental
	 */
	private void setDocumentFlowToUploaded(String id, String documentReference) throws DocumentaryFlowNotFoundException {

		final DocumentaryFlowModel documentaryFlowModel = repository.findById(id).orElseThrow(
				() -> new DocumentaryFlowNotFoundException(String.format("No se ha encontrado el flujo documental con id %s", id)));

		documentaryFlowModel.setDocumentReference(documentReference);
		documentaryFlowModel.setState(Status.UPLOADED);
		repository.save(documentaryFlowModel);
	}

	/**
	 * Cambia el estado del flujo documental con identificador "id" a "En tramitación"
	 *
	 * @param id
	 * 		identificador del modelo de flujo documental
	 *
	 * @throws DocumentaryFlowNotFoundException
	 * 		en el caso de que no se haya encontrado el flujo documental
	 */
	private void setDocumentFlowToInProcess(String id) throws DocumentaryFlowNotFoundException {

		final DocumentaryFlowModel documentaryFlowModel = repository.findById(id).orElseThrow(
				() -> new DocumentaryFlowNotFoundException(String.format("No se ha encontrado el flujo documental con id %s", id)));

		documentaryFlowModel.setState(Status.IN_PROCESS);
		repository.save(documentaryFlowModel);
	}

	/**
	 * Rellena los datos del registro del flujo documental con el registro asociado al documento tramitado y cambia el estado del flujo documental al que
	 * obtenga de Gestiona o a `UNKNOWN_ERROR` en el caso de que el documento no tenga anotaciones asociadas
	 *
	 * @param documentaryFlowModel
	 * 		datos del flujo documental
	 * @param documentReference
	 * 		referencia al recurso del documento en Gestiona
	 * @param documentProcessState
	 * 		estado de la tramitación del documento en Gestiona
	 */
	private void setDocumentFlowToFinalState(DocumentaryFlowModel documentaryFlowModel, String documentReference, DocumentProcessState documentProcessState) {

		final FileDocumentAnnotationModel annotation = gestionaAPIService.getFileDocumentAnnotation(documentReference);

		// En el caso de que no tenga una anotación asociada se informará su estado como `UNKNOWN_ERROR`
		Status documentaryFlowNewState = Status.UNKNOWN_ERROR;

		if (annotation != null) {
			documentaryFlowModel.setRegistryCodeGestiona(annotation.getCode());
			documentaryFlowModel.setRegistryIdGestiona(annotation.getId());

			documentaryFlowNewState = Status.valueOf(documentProcessState.toString());
		}

		documentaryFlowModel.setState(documentaryFlowNewState);

		repository.save(documentaryFlowModel);
	}

	/**
	 * Guarda el archivo subido en el directorio definido por la property "file.upload-path"
	 *
	 * @param file
	 * 		archivo a guardar
	 *
	 * @return la ruta absoluta del archivo guardado
	 *
	 * @throws IOException
	 * 		si se produce un error al guardar el archivo
	 */
	private String saveFile(MultipartFile file) throws IOException {

		final File directory = new File(uploadPath);
		if (!directory.exists()) {
			directory.mkdirs();
		}

		final File destinationFile = new File(uploadPath + File.separator + file.getOriginalFilename());
		log.info(destinationFile.getAbsolutePath());
		file.transferTo(destinationFile);

		return destinationFile.getAbsolutePath();

	}

	/**
	 * Obtiene el modelo del expediente a través de su identificador
	 *
	 * @param fileIdGestiona
	 * 		identificador del expediente en Gestiona
	 *
	 * @return el modelo del expediente
	 */
	private FileModel findFileModel(String fileIdGestiona) {
		List<FileModel> fileModelList = gestionaAPIService.getFileList();

		return fileModelList.stream()//
				.filter(file -> fileIdGestiona.equals(file.getId()))//
				.findFirst()//
				.orElseThrow(() -> new FileNotFoundException(String.format("No se ha encontrado el expediente con id %s", fileIdGestiona)));
	}

	/**
	 * Indica si el expediente está asignado a algún flujo documental
	 *
	 * @param fileModel
	 * 		el modelo del expediente
	 *
	 * @return true si y solo si existe un flujo documental con el expediente asignado, devuelve false en caso contrario
	 */
	private boolean hasDocumentaryFlow(FileModel fileModel) {
		return repository.findByFileIdGestiona(fileModel.getId()).isPresent();
	}

	/**
	 * Crea la petición de respuesta para descargar el documento con el contenido y el nombre indicados
	 *
	 * @param content
	 * 		contenido del documento a descargar
	 * @param fileName
	 * 		nombre del documento a descargar
	 *
	 * @return la respuesta con la descarga del documento
	 */
	private ResponseEntity<byte[]> createDocumentPetition(byte[] content, String fileName) {

		final HttpHeaders headers = new HttpHeaders();
		headers.setContentType(MediaType.APPLICATION_PDF);
		headers.setContentDispositionFormData("attachment", fileName);

		return new ResponseEntity<>(content, headers, HttpStatus.OK);
	}
}

