package main.gestiona.apirest.service;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;

import main.exception.GestionaAPIException;
import main.gestiona.apirest.GestionaAPIResources;

@Service
public class UploadService {

	private static final Logger log = LoggerFactory.getLogger(UploadService.class);

	@Autowired
	private GestionaAPIResources gestionaAPIResources;

	/**
	 * Crea un nuevo upload vacío en el recurso "uploadURL" y devuelve su URL.
	 *
	 * @param uploadURL
	 * 		url sobre la que se va a hacer la petición de tipo POST
	 *
	 * @return la URL del nuevo upload vacío
	 */
	public String createEmptyUpload(String uploadURL) {
		try {

			final ResponseEntity<String> jsonResponse = gestionaAPIResources.createPetition(uploadURL, HttpMethod.POST, null, null, String.class);

			return Objects.requireNonNull(jsonResponse.getHeaders().getLocation()).toString();

		} catch (Exception e) {
			throw new GestionaAPIException("Error al crear el upload vacío: " + e.getMessage());
		}
	}

	/**
	 * Sube el fichero al recurso "uploadURL" y devuelve el nombre del fichero subido.
	 *
	 * @param uploadURL
	 * 		url donde se va a subir el fichero
	 * @param documentPath
	 * 		path local del documento
	 *
	 * @return el nombre del fichero subido
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya subido el documento
	 */
	public String uploadDocument(String uploadURL, String documentPath) throws GestionaAPIException {

		try {
			// Creamos un FileSystemResource con el fichero a subir
			final File file = new File(documentPath);
			final FileSystemResource fileResource = new FileSystemResource(file);

			Map<String, String> extraHeaders = new HashMap<>();
			extraHeaders.put("Content-Length", String.valueOf(file.length()));

			final ResponseEntity<String> response = gestionaAPIResources.createPetition(uploadURL, HttpMethod.PUT, extraHeaders, fileResource, String.class);

			if (response.getStatusCode() != HttpStatus.OK) {
				throw new GestionaAPIException("Error al subir el documento: " + response.getStatusCode());
			}

			log.info("Documento subido correctamente a {}", uploadURL);
			return getFileNameWithoutExtension(file.getName());

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error al subir el documento: " + e.getMessage());
		}
	}

	/**
	 * Elimina la extensión del nombre de un archivo
	 *
	 * @param fileName
	 * 		nombre del archivo
	 *
	 * @return el nombre del archivo sin extensión
	 */
	private String getFileNameWithoutExtension(String fileName) {

		if (fileName == null || fileName.isEmpty()) {
			return "";
		}

		final int lastDotIndex = fileName.lastIndexOf('.');

		return (lastDotIndex == -1) ? fileName : fileName.substring(0, lastDotIndex);
	}

}
