package main.gestiona.apirest.service;

import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;

import main.connectors.thirds.view.service.GoogleSheetsService;
import main.exception.GestionaAPIException;
import main.gestiona.apirest.GestionaAPIResources;
import main.gestiona.apirest.GestionaAPIResources.NotificationChannel;
import main.gestiona.apirest.GestionaAPIResources.RelationType;
import main.gestiona.apirest.model.LinkModel;
import main.gestiona.apirest.model.ThirdAddressModel;
import main.gestiona.apirest.model.ThirdFilterModel;
import main.gestiona.apirest.model.ThirdModel;
import main.gestiona.apirest.model.ThirdOutputModel;
import main.gestiona.apirest.model.ThirdPageModel;
import main.gestiona.apirest.model.ThirdPartyModel;

@Service
public class ThirdService {

	private static final String FILTER_VIEW_FORMAT = "%s?filter-view=%s";
	private static final String MEDIA_TYPE_THIRD = "application/vnd.gestiona.third+json; version=3";
	private static final int SHEETS_NIF_COLUMN = 1;
	private static final int SHEETS_NAME_COLUMN = 2;
	private static final String GESTIONA_THIRDS_REL = "vnd.gestiona.thirds";
	private static final String THIRD_REL = "third";
	private static final String SELF_REL = "self";
	private static final String NIF_TYPE = "NIF";
	private static final String TELEMATIC_TYPE = "TELEMATIC";
	private static final String SPAIN_CODE = "ESP";
	private static final String PHISIC_TYPE = "PHISIC";
	private static final String MEDIA_TYPE_THIRD_PARTY = "application/vnd.gestiona.thirdparty+json; version=3";
	private static final Logger log = LoggerFactory.getLogger(ThirdService.class);

	private final GestionaAPIResources gestionaAPIResources;

	@Autowired
	private GoogleSheetsService googleSheetsService;

	@Autowired
	public ThirdService(GestionaAPIResources gestionaAPIResources) {

		this.gestionaAPIResources = gestionaAPIResources;
	}

	/**
	 * Obtiene el modelo de un tercero dado su nif
	 *
	 * @param nif
	 * 		el dni del tercero a buscar
	 *
	 * @return el modelo del tercero en el caso de que lo haya encontrado
	 *
	 * @throws GestionaAPIException
	 * 		en el caso de que haya ocurrido un error al realizar la petición o que no se haya encontrado al tercero
	 */
	public ThirdModel getThirdByNif(String nif) throws GestionaAPIException {
		try {

			final String url = gestionaAPIResources.getBookmarkUrl(GESTIONA_THIRDS_REL);

			final ThirdFilterModel thirdFilterModel = new ThirdFilterModel(nif);

			final String encodedUrlSafe = Base64.getUrlEncoder().encodeToString(thirdFilterModel.toString().getBytes(StandardCharsets.UTF_8));

			final String urlWithParam = String.format(FILTER_VIEW_FORMAT, url, encodedUrlSafe);

			final ResponseEntity<ThirdPageModel> thirdByNif = gestionaAPIResources.createPetition(urlWithParam, HttpMethod.GET, null, null,
					ThirdPageModel.class);

			if (thirdByNif.getStatusCode().equals(HttpStatus.NO_CONTENT)) {
				throw new GestionaAPIException("No se ha encontrado ningún tercero con el NIF: " + nif);
			}

			return Objects.requireNonNull(thirdByNif.getBody()).getContent().getFirst();

		} catch (Exception e) {
			throw new GestionaAPIException("Error obteniendo el modelo del tercero: " + e.getMessage());
		}
	}

	/**
	 * Crea el tercero en Gestiona a través de los datos de la hoja de Google Sheets
	 *
	 * @param nif
	 * 		nif del tercero a crear en Gestiona
	 *
	 * @throws GestionaAPIException
	 * 		en el caso de que ocurra algún error en la creación del tercero
	 */
	public void createThird(String nif) throws GestionaAPIException {

		try {
			final ThirdModel third = createThirdModel(nif);
			final String url = gestionaAPIResources.getBookmarkUrl(GESTIONA_THIRDS_REL);

			final Map<String, String> extraHeaders = new HashMap<>();
			extraHeaders.put(GestionaAPIResources.CONTENT_TYPE, MEDIA_TYPE_THIRD);

			final ResponseEntity<ThirdModel> newGestionaThird = gestionaAPIResources.createPetition(url, HttpMethod.POST, extraHeaders, third,
					ThirdModel.class);

			if (!newGestionaThird.getStatusCode().equals(HttpStatus.CREATED)) {
				throw new GestionaAPIException("No se ha podido crear el tercero con NIF: " + nif);
			}

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error al crear el tercero: " + e.getMessage());
		}
	}

	/**
	 * Crea el modelo de salida del tercero para añadirlo al listado del circuito de tramitación
	 *
	 * @param thirdModel
	 * 		modelo del tercero
	 *
	 * @return el modelo de salida del tercero
	 */
	public ThirdOutputModel createThirdOutputModel(ThirdModel thirdModel) {
		final ThirdOutputModel thirdOutputModel = new ThirdOutputModel();

		final NotificationChannel notificationChannel = thirdModel.getNotification_channel();
		thirdOutputModel.setNotification_channel(notificationChannel);
		thirdOutputModel.setRelation(RelationType.INVOLVED);

		final LinkModel thirdLink = new LinkModel(THIRD_REL, GestionaAPIResources.obtainLinkByRel(thirdModel.getLinks(), SELF_REL));
		thirdOutputModel.addLink(thirdLink);

		return thirdOutputModel;
	}

	/**
	 * Crea el modelo de salida del tercero a través del modelo del tercero
	 *
	 * @param thirdModel
	 * 		modelo del tercero
	 *
	 * @return modelo de salida del tercero
	 */
	public ThirdPartyModel createThirdPartyModel(ThirdModel thirdModel) {
		return new ThirdPartyModel(thirdModel);
	}

	/**
	 * Añade el tercero al registro en Gestiona
	 *
	 * @param url
	 * 		url sobre la que hacer la petición
	 * @param thirdPartyModel
	 * 		modelo del tercero a añadir
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API de Gestiona
	 */
	public void addThirdPartyModel(String url, ThirdPartyModel thirdPartyModel) throws GestionaAPIException {
		try {
			final Map<String, String> extraHeaders = new HashMap<>();
			extraHeaders.put(GestionaAPIResources.CONTENT_TYPE, MEDIA_TYPE_THIRD_PARTY);

			final ResponseEntity<ThirdPartyModel> newThirdParty = gestionaAPIResources.createPetition(url, HttpMethod.POST, extraHeaders, thirdPartyModel,
					ThirdPartyModel.class);

			if (!newThirdParty.getStatusCode().equals(HttpStatus.CREATED)) {
				throw new GestionaAPIException(String.format("Error al añadir el tercero al registro con url %s: ", url));
			}
		} catch (RestClientException e) {
			throw new GestionaAPIException(String.format("Error al añadir el tercero al registro con url %s: ", url) + e.getMessage());
		}
	}

	/**
	 * Añade los atributos necesarios al modelo de salida del tercero con canal de notificación `PAPER`
	 *
	 * @param thirdOutputModel
	 * 		modelo de salida del tercero
	 * @param thirdAddressModel
	 * 		modelo de la dirección del tercero
	 */
	public void addPaperNecessaryFields(ThirdOutputModel thirdOutputModel, ThirdAddressModel thirdAddressModel) {

		thirdOutputModel.setAddress(thirdAddressModel.getAddress());
		thirdOutputModel.setCountry(thirdAddressModel.getCountry());
		thirdOutputModel.setZip(thirdAddressModel.getZip_code());
		thirdOutputModel.setZone(thirdAddressModel.getZone());

	}

	/**
	 * Obtiene el modelo de la dirección del tercero
	 *
	 * @param url
	 * 		url sobre la cual realizar la petición
	 *
	 * @return el modelo de la dirección del tercero
	 *
	 * @throws GestionaAPIException
	 * 		en el caso de que haya ocurrido un error con la petición o que np se haya encontrado el modelo
	 */
	public ThirdAddressModel getThirdAddressModel(String url) throws GestionaAPIException {
		try {

			return Objects.requireNonNull(gestionaAPIResources.createPetition(url, HttpMethod.GET, null, null, ThirdAddressModel.class).getBody());

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error obteniendo la dirección del tercero: " + e.getMessage());
		}
	}

	/**
	 * Crea el modelo del tercero a través de los datos obtenidos de la hoja de Google Sheets
	 *
	 * @param nif
	 * 		nif del tercero a crear
	 *
	 * @return el modelo del tercero contribuido
	 */
	private ThirdModel createThirdModel(String nif) {
		final List<Object> thirdData = googleSheetsService.getThirdByNIF(nif);

		final ThirdModel newThird = new ThirdModel();
		newThird.setNif(thirdData.get(SHEETS_NIF_COLUMN).toString());
		newThird.setFull_name(thirdData.get(SHEETS_NAME_COLUMN).toString());
		newThird.setNif_type(NIF_TYPE);
		newThird.setNif_country(SPAIN_CODE);
		newThird.setType(PHISIC_TYPE);
		newThird.setNotification_channel(TELEMATIC_TYPE);

		final String[] splitName = thirdData.get(SHEETS_NAME_COLUMN).toString().split(" ");
		newThird.setFirst_name(splitName[0]);
		newThird.setFirst_surname(splitName[1]);

		if (splitName[2] != null) {
			newThird.setSecond_surname(splitName[2]);
		}

		return newThird;
	}

}
