package main.gestiona.apirest.service;

import static main.gestiona.apirest.GestionaAPIResources.CONTENT_TYPE;
import static main.gestiona.apirest.GestionaAPIResources.SELF_REL;
import static main.gestiona.apirest.GestionaAPIResources.obtainLinkByRel;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;

import main.exception.GestionaAPIException;
import main.gestiona.apirest.GestionaAPIResources;
import main.gestiona.apirest.config.RegistriesConfig;
import main.gestiona.apirest.model.FileDocumentAnnotationModel;
import main.gestiona.apirest.model.FileModel;
import main.gestiona.apirest.model.LinksModel;
import main.gestiona.apirest.model.RegistryAnnotationModel;
import main.gestiona.apirest.model.RegistryOfficeModel;
import main.gestiona.apirest.model.RegistryOfficesListModel;

@Service
public class RegistryService {

	private static final Logger log = LoggerFactory.getLogger(RegistryService.class);

	private static final String GESTIONA_REGISTRY_OFFICES_REL = "vnd.gestiona.registry.offices";
	private static final String REGISTRY_ANNOTATION_MEDIA_TYPE = "application/vnd.gestiona.registry-annotation+json;version=2";
	private static final String ANNOTATION_REL = "annotation";
	private static final String FINALIZE_REL = "finalize";
	private static final String FILE_REL = "file";
	private static final String MEDIA_TYPE_LINKS = "application/vnd.gestiona.links+json";
	private static final String SEND_TO_FILE_REL = "send-to-file";

	private final String registryOfficeCode;
	private final String registryCategory;
	private final String registrySubcategory;

	@Autowired
	private GestionaAPIResources gestionaAPIResources;

	public RegistryService(RegistriesConfig registriesConfig) {
		this.registryOfficeCode = registriesConfig.getRegistryOfficeCode();
		this.registryCategory = registriesConfig.getCategoryName();
		this.registrySubcategory = registriesConfig.getSubcategoryName();
	}

	/**
	 * Crea un nuevo registro de entrada en la API de Gestiona y devuelve el código del nuevo registro.
	 *
	 * @param subject
	 * 		asunto del registro
	 *
	 * @return el código del nuevo registro creado en Gestiona
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya creado el registro
	 */
	public RegistryAnnotationModel createInputRegistry(String subject) throws GestionaAPIException {

		try {

			// Saca la url de la oficina de registro buscada
			final String url = obtainInputRegistryOfficeLink(getRegistryOffices());

			// Crea un nuevo registro de entrada con la información dada
			final RegistryAnnotationModel inputRegistryModel = new RegistryAnnotationModel(subject, registryCategory, registrySubcategory);

			final Map<String, String> extraHeaders = new HashMap<>();
			extraHeaders.put(CONTENT_TYPE, REGISTRY_ANNOTATION_MEDIA_TYPE);

			final ResponseEntity<String> response = gestionaAPIResources.createPetition(url, HttpMethod.POST, extraHeaders, inputRegistryModel, String.class);

			// Comprueba que se haya creado correctamente
			if (response.getStatusCode() == HttpStatus.CREATED) {
				final String locationUrl = Objects.requireNonNull(response.getHeaders().getLocation()).toString();

				log.info("Se ha creado el registro de entrada en la dirección {}", locationUrl);
				return obtainRegistryAnnotationModel(locationUrl);
			}

			throw new GestionaAPIException(String.format("No se ha podido crear el registro de entrada: %s", response.getStatusCode()));

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error al crear el registro de entrada: " + e.getMessage());
		}

	}

	/**
	 * Obtiene el código del registro de entrada a partir de su URL
	 *
	 * @param inputRegistryLocation
	 * 		URL del registro de entrada
	 *
	 * @return el código del registro
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API
	 */
	public RegistryAnnotationModel obtainRegistryAnnotationModel(String inputRegistryLocation) throws GestionaAPIException {
		try {

			return Objects.requireNonNull(
					gestionaAPIResources.createPetition(inputRegistryLocation, HttpMethod.GET, null, null, RegistryAnnotationModel.class).getBody());

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error obteniendo el registro de entrada: " + e.getMessage());
		}
	}

	/**
	 * Obtiene la dirección de la oficina de registro cuyo código coincide con la property gestiona.api-registry-office-code
	 *
	 * @param registryOfficeModelList
	 * 		el listado de las oficinas de registro
	 *
	 * @return la dirección de la oficina de registro
	 *
	 * @throws GestionaAPIException
	 * 		si no se ha encontrado la oficina de registro
	 */
	public String obtainInputRegistryOfficeLink(List<RegistryOfficeModel> registryOfficeModelList) throws GestionaAPIException {

		final RegistryOfficeModel registryOfficeModel = registryOfficeModelList.stream()//
				.filter(office -> office.getCode().equals(registryOfficeCode))//
				.findFirst()//
				.orElseThrow(() -> new GestionaAPIException("Oficina de registro no encontrada"));

		return registryOfficeModel.getLinks().stream()//
				.filter(link -> link.getRel().equalsIgnoreCase("input"))//
				.findFirst()//
				.orElseThrow(() -> new GestionaAPIException("Enlace input no encontrado"))//
				.getHref();
	}

	/**
	 * Obtiene la lista de oficinas de registro de la API de Gestiona
	 *
	 * @return el listado de oficinas de registro mapeadas a RegistryOfficeModel
	 *
	 * @throws GestionaAPIException
	 * 		en el caso de que ocurra algún error en la petición a la API de Gestiona
	 */
	public List<RegistryOfficeModel> getRegistryOffices() throws GestionaAPIException {
		try {

			final String url = gestionaAPIResources.getBookmarkUrl(GESTIONA_REGISTRY_OFFICES_REL);

			return Objects.requireNonNull(gestionaAPIResources.createPetition(url, HttpMethod.GET, null, null, RegistryOfficesListModel.class).getBody())
					.getContent();

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error obteniendo la lista de oficinas de registro: " + e.getMessage());
		}
	}

	/**
	 * Realiza la petición sobre la API de Gestiona para obtener el modelo de la anotación
	 *
	 * @param fileDocumentAnnotationModel
	 * 		modelo de la anotación asociada al documento
	 *
	 * @return modelo de la anotación
	 *
	 * @throws GestionaAPIException
	 * 		en el caso de que ocurra algún error en la petición a la API de Gestiona
	 */
	public RegistryAnnotationModel getRegistryAnnotationModel(FileDocumentAnnotationModel fileDocumentAnnotationModel) throws GestionaAPIException {
		try {

			final String url = obtainLinkByRel(fileDocumentAnnotationModel.getLinks(), ANNOTATION_REL);

			return Objects.requireNonNull(gestionaAPIResources.createPetition(url, HttpMethod.GET, null, null, RegistryAnnotationModel.class).getBody());

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error obteniendo el registro de la anotación: " + e.getMessage());
		}
	}

	/**
	 * Finaliza el registro de entrada
	 *
	 * @param registryAnnotationModel
	 * 		modelo del registro de entrada
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la petición a la API de Gestiona
	 */
	public void finalizeRegistryAnnotation(RegistryAnnotationModel registryAnnotationModel) throws GestionaAPIException {
		try {

			final String url = obtainLinkByRel(registryAnnotationModel.getLinks(), FINALIZE_REL);

			final ResponseEntity<String> response = gestionaAPIResources.createPetition(url, HttpMethod.POST, null, null, String.class);

			if (response.getStatusCode() != HttpStatus.OK) {
				throw new GestionaAPIException(String.format("No se ha podido finalizar el registro: %s", registryAnnotationModel.getCode()));
			}

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error finalizando el registro de la anotación: " + e.getMessage());
		}
	}

	/**
	 * Asigna un registro de entrada a un expediente
	 *
	 * @param fileModel
	 * 		modelo del expediente al que asignarle el registro
	 * @param registryAnnotationModel
	 * 		modelo del registro de entrada a asignar
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la petición a la API de Gestiona
	 */
	public void addToFile(FileModel fileModel, RegistryAnnotationModel registryAnnotationModel) throws GestionaAPIException {

		try {
			final String fileReference = obtainLinkByRel(fileModel.getLinks(), SELF_REL);

			final LinksModel linksModel = new LinksModel(FILE_REL, fileReference);

			final Map<String, String> extraHeaders = new HashMap<>();
			extraHeaders.put(CONTENT_TYPE, MEDIA_TYPE_LINKS);

			final String url = obtainLinkByRel(registryAnnotationModel.getLinks(), SEND_TO_FILE_REL);

			final ResponseEntity<String> response = gestionaAPIResources.createPetition(url, HttpMethod.POST, extraHeaders, linksModel, String.class);

			if (response.getStatusCode() != HttpStatus.OK) {
				throw new GestionaAPIException(
						String.format("No se ha podido añadir el registro %s al expediente %s", registryAnnotationModel.getCode(), fileModel.getCode()));
			}

		} catch (RestClientException e) {
			throw new GestionaAPIException(
					String.format("No se ha podido añadir el registro %s al expediente %s:", registryAnnotationModel.getCode(), fileModel.getCode()) +
							e.getMessage());
		}
	}

}
