package main.gestiona.apirest.service;

import static main.gestiona.apirest.GestionaAPIResources.obtainLinkByRel;

import java.util.Objects;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;

import main.exception.GestionaAPIException;
import main.gestiona.apirest.GestionaAPIResources;
import main.gestiona.apirest.model.ExternalProcedureModel;
import main.gestiona.apirest.model.ExternalProcedurePageModel;
import main.gestiona.apirest.model.FileOpeningModel;
import main.gestiona.apirest.model.ProcedureModel;
import main.gestiona.apirest.model.ProcedurePageModel;

@Service
public class ProcedureService {

	private static final String NEW_CATALOG_REL = "vnd.gestiona.catalog-2015";
	private static final String CREATE_FILE_REL = "create-file";

	@Autowired
	GestionaAPIResources gestionaAPIResources;

	/**
	 * Obtiene el modelo de un procedimiento por su nombre
	 *
	 * @param procedureName
	 * 		nombre del procedimiento a buscar
	 *
	 * @return modelo del procedimiento con el nombre especificado
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya encontrado el procedimiento
	 */
	public ProcedureModel getProcedure(String procedureName) throws GestionaAPIException {
		try {

			final String url = gestionaAPIResources.getBookmarkUrl(NEW_CATALOG_REL);

			final ProcedurePageModel procedurePageModel = Objects.requireNonNull(
					gestionaAPIResources.createPetition(url, HttpMethod.GET, null, null, ProcedurePageModel.class).getBody());

			return searchProcedureRecursive(procedurePageModel, procedureName);

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error al obtener el procedimiento: " + procedureName + e);
		}
	}

	/**
	 * Crea un expediente con el trámite externo `externalProcedureModel`
	 *
	 * @param externalProcedureModel
	 * 		modelo del trámite externo
	 *
	 * @return modelo del file opening creado
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API
	 */
	public FileOpeningModel createFile(ExternalProcedureModel externalProcedureModel) throws GestionaAPIException {
		try {

			final String url = obtainLinkByRel(externalProcedureModel.getLinks(), CREATE_FILE_REL);

			return gestionaAPIResources.createPetition(url, HttpMethod.POST, null, null, FileOpeningModel.class).getBody();

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error al crear el expediente con el trámite externo: " + externalProcedureModel.getTitle() + e);
		}
	}

	/**
	 * Obtiene el modelo de un trámite externo por su nombre
	 *
	 * @param externalProceduresUrl
	 * 		url del recurso de trámites externos
	 * @param externalProcedureTitle
	 * 		título del trámite externo a buscar
	 *
	 * @return modelo del trámite externo con el nombre especificado
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya encontrado el trámite externo
	 */
	public ExternalProcedureModel getExternalProcedureUrl(String externalProceduresUrl, String externalProcedureTitle) throws GestionaAPIException {
		try {

			final ExternalProcedurePageModel procedurePageModel = Objects.requireNonNull(
					gestionaAPIResources.createPetition(externalProceduresUrl, HttpMethod.GET, null, null, ExternalProcedurePageModel.class).getBody());

			return searchExternalProcedureRecursive(procedurePageModel, externalProcedureTitle);

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error al obtener el procedimiento: " + externalProcedureTitle + e);
		}
	}

	/**
	 * Búsqueda recursiva para encontrar un procedimiento en las páginas obtenidas
	 *
	 * @param procedurePageModel
	 * 		modelo de la página de procedimientos en la que se va a realizar la búsqueda
	 * @param procedureName
	 * 		nombre del procedimiento a buscar
	 *
	 * @return el modelo del procedimiento si lo encuentra, o una excepción en caso contrario
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya encontrado el procedimiento
	 */
	private ProcedureModel searchProcedureRecursive(ProcedurePageModel procedurePageModel, String procedureName) throws GestionaAPIException {
		try {

			final Optional<ProcedureModel> procedureModel = procedurePageModel.getContent().stream().filter(
					procedure -> procedure.getName().equals(procedureName)).findFirst();

			// Caso base 1: encuentra el procedimiento en la página actual
			if (procedureModel.isPresent()) {
				return procedureModel.get();
			}
			// Caso recursivo: sigue a la siguiente página hasta encontrar el procedimiento o llegar a una página vacía
			if (procedurePageModel.getLinks().stream().anyMatch(link -> link.getRel().equals(GestionaAPIResources.NEXT_PAGE_REL))) {

				final ProcedurePageModel nextPageProcedureModel = (ProcedurePageModel) gestionaAPIResources.getNextPage(procedurePageModel,
						ProcedurePageModel.class);

				return searchProcedureRecursive(nextPageProcedureModel, procedureName);
			}
			// Caso base 2: No ha encontrado el procedimiento en ninguna página
			throw new GestionaAPIException(String.format("No se ha encontrado el procedimiento: %s", procedureName));

		} catch (Exception e) {
			throw new GestionaAPIException("Error al obtener el procedimiento: " + procedureName + e);
		}
	}

	/**
	 * Búsqueda recursiva para encontrar un trámite externo en las páginas obtenidas
	 *
	 * @param procedurePageModel
	 * 		modelo de la página de trámites externos en la que se va a realizar la búsqueda
	 * @param procedureTitle
	 * 		título del trámite externo a buscar
	 *
	 * @return modelo del trámite externo si lo encuentra, o una excepción en caso contrario
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya encontrado el trámite externo
	 */
	private ExternalProcedureModel searchExternalProcedureRecursive(ExternalProcedurePageModel procedurePageModel, String procedureTitle)
			throws GestionaAPIException {
		try {
			// Busca el procedimiento externo en la página actual
			final Optional<ExternalProcedureModel> procedureModel = procedurePageModel.getContent().stream().filter(
					procedure -> procedure.getTitle().equals(procedureTitle)).findFirst();

			// Caso base 1: encuentra el trámite externo en la página actual
			if (procedureModel.isPresent()) {
				return procedureModel.get();
			}

			// Caso recursivo: sigue a la siguiente página hasta encontrar el trámite externo o llegar a una página vacía
			if (procedurePageModel.getLinks().stream().anyMatch(link -> link.getRel().equals(GestionaAPIResources.NEXT_PAGE_REL))) {

				final ExternalProcedurePageModel nextPageProcedureModel = (ExternalProcedurePageModel) gestionaAPIResources.getNextPage(procedurePageModel,
						ExternalProcedurePageModel.class);

				return searchExternalProcedureRecursive(nextPageProcedureModel, procedureTitle);
			}

			// Caso base 2: No ha encontrado el procedimiento en ninguna página
			throw new GestionaAPIException(String.format("No se ha encontrado el procedimiento: %s", procedureTitle));

		} catch (Exception e) {
			throw new GestionaAPIException("Error al obtener el procedimiento: " + procedureTitle + e);
		}
	}
}
