package main.gestiona.apirest.service;

import static main.gestiona.apirest.GestionaAPIResources.CONTENT_TYPE;
import static main.gestiona.apirest.GestionaAPIResources.obtainLinkByRel;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;

import main.exception.GestionaAPIException;
import main.gestiona.apirest.GestionaAPIResources;
import main.gestiona.apirest.model.CircuitTemplateModel;
import main.gestiona.apirest.model.DocumentModel;
import main.gestiona.apirest.model.DocumentProcessStateModel;
import main.gestiona.apirest.model.FileDocumentAnnotationModel;
import main.gestiona.apirest.model.FileDocumentAnnotationPageModel;
import main.gestiona.apirest.model.LinkModel;

@Service
public class DocumentService {

	private static final String GESTIONA_FILE_DOCUMENT_MEDIA_TYPE = "application/vnd.gestiona.file-document+json; version=3";
	private static final String CIRCUITS_TEMPLATE_MEDIA_TYPE = "application/vnd.gestiona.circuits.template-filedoc+json; version=5";
	private static final String CIRCUIT_REL = "circuit";
	private static final String DOCUMENT_ANNOTATIONS_REL = "document-annotations";
	private static final String CONTENT_REL = "content";

	private static final Logger log = LoggerFactory.getLogger(DocumentService.class);

	@Autowired
	private GestionaAPIResources gestionaAPIResources;

	/**
	 * Crea un nuevo documento en el expediente con código "fileCode" con el fichero subido en "uploadURL" y devuelve la referencia al documento creado.
	 *
	 * @param documentsAndFoldersURL
	 * 		url sobre la que se va a hacer la petición de tipo POST
	 * @param uploadURL
	 * 		url donde se ha subido el fichero del documento
	 * @param fileName
	 * 		nombre del fichero subido en "uploadURL"
	 *
	 * @return el modelo del documento creado
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya creado el documento
	 */
	public String createDocument(String documentsAndFoldersURL, String uploadURL, String fileName) throws GestionaAPIException {

		try {
			// Creamos un DocumentModel vacío con el link al contenido que hemos subido
			final List<LinkModel> links = new ArrayList<>();
			links.add(new LinkModel(CONTENT_REL, uploadURL));

			final DocumentModel newDocument = new DocumentModel(links, fileName);

			final Map<String, String> extraHeaders = new HashMap<>();
			extraHeaders.put(GestionaAPIResources.CONTENT_TYPE, GESTIONA_FILE_DOCUMENT_MEDIA_TYPE);

			final DocumentModel documentModel = gestionaAPIResources.createPetition(documentsAndFoldersURL, HttpMethod.POST, extraHeaders, newDocument,
					DocumentModel.class).getBody();

			if (documentModel == null) {
				throw new GestionaAPIException("Error al crear el documento");
			}

			return obtainLinkByRel(documentModel.getLinks(), "self");

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error al crear el documento: " + e.getMessage());
		}
	}

	/**
	 * Obtiene el modelo del documento a partir de su URL
	 *
	 * @param documentURL
	 * 		la URL al recurso donde se encuentra la carpeta
	 *
	 * @return el modelo del documento
	 *
	 * @throws GestionaAPIException
	 * 		si no se ha podido obtener el modelo del documento
	 */
	public DocumentModel obtainDocument(String documentURL) throws GestionaAPIException {
		try {

			log.info("URI del documento: {}", documentURL);
			return gestionaAPIResources.createPetition(documentURL, HttpMethod.GET, null, null, DocumentModel.class).getBody();

		} catch (RestClientException e) {
			throw new GestionaAPIException("No se ha podido obtener el documento: " + e.getMessage());
		}
	}

	/**
	 * Procesa el documento con el código "documentCode" y con el circuito de tramitación "circuitTemplateModel".
	 *
	 * @param documentReference
	 * 		modelo del documento a tramitar
	 * @param circuitTemplateModel
	 * 		modelo del circuito de tramitación con el que se va a tramitar el documento
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya tramitado el documento
	 */
	public void processDocument(String documentReference, CircuitTemplateModel circuitTemplateModel) throws GestionaAPIException {

		final DocumentModel documentModel = obtainDocument(documentReference);
		final String processDocumentURL = obtainLinkByRel(documentModel.getLinks(), CIRCUIT_REL);

		try {
			final Map<String, String> extraHeaders = new HashMap<>();
			extraHeaders.put(CONTENT_TYPE, CIRCUITS_TEMPLATE_MEDIA_TYPE);

			log.info("Se va a tramitar el documento con referencia: {}", processDocumentURL);
			log.info("Circuito elegido: {}", circuitTemplateModel);

			final ResponseEntity<String> response = gestionaAPIResources.createPetition(processDocumentURL, HttpMethod.POST, extraHeaders, circuitTemplateModel,
					String.class);

			if (response.getStatusCode() != HttpStatus.CREATED) {
				throw new GestionaAPIException("No se ha podido tramitar el documento: " + response.getStatusCode());
			}

			log.info("Se ha mandado a tramitar el documento con éxito");

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error al tramitar el documento: " + e.getMessage());
		}

	}

	/**
	 * Obtiene el estado de tramitación del documento con referencia en Gestiona "documentReference".
	 *
	 * @param documentReference
	 * 		referencia al documento en Gestiona
	 *
	 * @return el estado de tramitación del documento mapeado a un DocumentProcessStateModel
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya obtenido el estado de tramitación
	 */
	public DocumentProcessStateModel checkDocumentProcessStatus(String documentReference) throws GestionaAPIException {

		try {

			final DocumentModel documentModel = obtainDocument(documentReference);
			final String processStatusURL = obtainLinkByRel(documentModel.getLinks(), CIRCUIT_REL);

			return gestionaAPIResources.createPetition(processStatusURL, HttpMethod.GET, null, null, DocumentProcessStateModel.class).getBody();

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error al obtener el estado de tramitación del documento: " + e.getMessage());
		}
	}

	/**
	 * Obtiene el listado de anotaciones del documento con referencia `documentReference` en Gestiona
	 *
	 * @param documentReference
	 * 		referencia al documento en Gestiona
	 *
	 * @return el listado de anotaciones del documento
	 */
	public List<FileDocumentAnnotationModel> getDocumentAnnotations(String documentReference) {
		try {

			final DocumentModel documentModel = obtainDocument(documentReference);
			final String processStatusURL = obtainLinkByRel(documentModel.getLinks(), DOCUMENT_ANNOTATIONS_REL);

			final FileDocumentAnnotationPageModel fileDocumentAnnotationPageModel = gestionaAPIResources.createPetition(processStatusURL, HttpMethod.GET, null,
					null, FileDocumentAnnotationPageModel.class).getBody();

			return (fileDocumentAnnotationPageModel != null) ? fileDocumentAnnotationPageModel.getContent() : null;

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error al obtener el el listado de anotaciones del documento: " + e.getMessage());
		}
	}

	/**
	 * Obtiene el contenido de un documento en Gestiona
	 *
	 * @param documentModel
	 * 		datos del documento
	 *
	 * @return contenido del documento
	 */
	public byte[] getDocumentContent(DocumentModel documentModel) {
		try {
			final String url = obtainLinkByRel(documentModel.getLinks(), CONTENT_REL);

			return gestionaAPIResources.createPetition(url, HttpMethod.GET, null, null, byte[].class).getBody();

		} catch (RestClientException e) {
			throw new GestionaAPIException("Error al obtener el contenido del documento: " + e.getMessage());
		}
	}

}
