package main.gestiona.apirest;

import static main.gestiona.apirest.GestionaAPIResources.DOCUMENTS_AND_FOLDERS_REL;
import static main.gestiona.apirest.GestionaAPIResources.obtainLinkByRel;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import main.exception.GestionaAPIException;
import main.gestiona.apirest.GestionaAPIResources.NotificationChannel;
import main.gestiona.apirest.model.CircuitTemplateModel;
import main.gestiona.apirest.model.DocumentModel;
import main.gestiona.apirest.model.DocumentProcessStateModel;
import main.gestiona.apirest.model.ExternalProcedureModel;
import main.gestiona.apirest.model.FileDocumentAnnotationModel;
import main.gestiona.apirest.model.FileModel;
import main.gestiona.apirest.model.FileOpeningModel;
import main.gestiona.apirest.model.ProcedureModel;
import main.gestiona.apirest.model.RegistryAnnotationModel;
import main.gestiona.apirest.model.ThirdAddressModel;
import main.gestiona.apirest.model.ThirdModel;
import main.gestiona.apirest.model.ThirdOutputModel;
import main.gestiona.apirest.model.ThirdPartyModel;
import main.gestiona.apirest.service.CircuitService;
import main.gestiona.apirest.service.DocumentService;
import main.gestiona.apirest.service.FileService;
import main.gestiona.apirest.service.ProcedureService;
import main.gestiona.apirest.service.RegistryService;
import main.gestiona.apirest.service.ThirdService;
import main.gestiona.apirest.service.UploadService;

@Service
public class GestionaAPIService {

	private static final Logger log = LoggerFactory.getLogger(GestionaAPIService.class);

	private static final String GESTIONA_UPLOADS_REL = "vnd.gestiona.uploads";
	private static final String DEFAULT_ADDRESS = "default-address";
	private static final String THIRD_PARTIES_REL = "thirdparties";
	private static final String EXTERNAL_PROCEDURES_REL = "external-procedures";
	private static final String CIRCUITS_TEMPLATES_REL = "circuits-templates";

	@Autowired
	private GestionaAPIResources gestionaAPIResources;

	@Autowired
	private DocumentService documentService;

	@Autowired
	private UploadService uploadService;

	@Autowired
	private RegistryService registryService;

	@Autowired
	private CircuitService circuitService;

	@Autowired
	private FileService fileService;

	@Autowired
	private ThirdService thirdService;

	@Autowired
	private ProcedureService procedureService;

	/**
	 * Extrae de la API de Gestiona los expedientes asociados al usuario del access token. Extrae únicamente la primera página de resultados.
	 *
	 * @return el listado de expedientes mapeados a la clase FileModel.
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API
	 */
	public List<FileModel> getFileList() throws GestionaAPIException {

		return fileService.getFileList();
	}

	/**
	 * Crea un nuevo expediente con el trámite externo "externalProcedureName" del procedimiento "procedureName"
	 *
	 * @param procedureName
	 * 		nombre del procedimiento
	 * @param externalProcedureTitle
	 * 		título del trámite externo
	 *
	 * @return el modelo del expediente creado
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error al crear el expediente
	 */
	public FileModel createFileWithProcedure(String procedureName, String externalProcedureTitle) throws GestionaAPIException {

		final ProcedureModel procedureModel = procedureService.getProcedure(procedureName);

		final ExternalProcedureModel externalProcedure = procedureService.getExternalProcedureUrl(
				obtainLinkByRel(procedureModel.getLinks(), EXTERNAL_PROCEDURES_REL), externalProcedureTitle);

		final FileOpeningModel fileOpeningModel = procedureService.createFile(externalProcedure);
		return fileService.openFile(fileOpeningModel);
	}

	/**
	 * Asigna un registro de entrada a un expediente
	 *
	 * @param fileModel
	 * 		modelo del expediente
	 * @param registryAnnotationModel
	 * 		modelo del registro de entrada a asignar al expediente
	 *
	 * @throws GestionaAPIException
	 * 		en caso de que no se pueda asignar el registro al expediente
	 */
	public void addRegistryToFile(FileModel fileModel, RegistryAnnotationModel registryAnnotationModel) throws GestionaAPIException {
		registryService.addToFile(fileModel, registryAnnotationModel);
	}

	/**
	 * Obtiene de nuevo el modelo de un registro de entrada
	 *
	 * @param registryAnnotationModel
	 * 		modelo del registro de entrada a obtener de nuevo
	 *
	 * @return el modelo del registro de entrada actualizado
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API
	 */
	public RegistryAnnotationModel refreshRegistryAnnotationModel(RegistryAnnotationModel registryAnnotationModel) throws GestionaAPIException {
		final String registryAnnotationReference = obtainLinkByRel(registryAnnotationModel.getLinks(), GestionaAPIResources.SELF_REL);
		return registryService.obtainRegistryAnnotationModel(registryAnnotationReference);
	}

	/**
	 * Crea un nuevo registro de entrada en la API de Gestiona y devuelve el código del nuevo registro.
	 *
	 * @param subject
	 * 		asunto del registro de entrada
	 *
	 * @return el código del nuevo registro creado en Gestiona
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya creado el registro
	 */
	public RegistryAnnotationModel createInputRegistry(String subject) throws GestionaAPIException {

		return registryService.createInputRegistry(subject);
	}

	/**
	 * Obtiene el listado de circuitos de tramitación disponibles para el documento asociado a la referencia
	 *
	 * @param documentReference
	 * 		referencia al documento en Gestiona
	 *
	 * @return el listado de circuitos de tramitación
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se encuentre alguno de los datos en esta
	 */
	public List<CircuitTemplateModel> getCircuitTemplates(String documentReference) throws GestionaAPIException {

		// Devolvemos el listado de circuitos de tramitación asociados al documento
		final DocumentModel documentModel = documentService.obtainDocument(documentReference);
		return circuitService.obtainCircuitsTemplates(obtainLinkByRel(documentModel.getLinks(), CIRCUITS_TEMPLATES_REL));
	}

	/**
	 * Añade un documento al expediente con código "fileCode" y devuelve el modelo del documento añadido.
	 *
	 * @param fileModel
	 * 		modelo del expediente a añadirle el documento
	 * @param documentPath
	 * 		path local del documento
	 *
	 * @return el modelo del documento añadido
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya añadido el documento
	 */
	public String addDocumentToFile(FileModel fileModel, String documentPath) throws GestionaAPIException {

		// Sacamos el recurso "document-and-folders" del expediente
		final String documentsAndFoldersURL = obtainLinkByRel(fileModel.getLinks(), DOCUMENTS_AND_FOLDERS_REL);

		// Sacamos el bookmark de uploads
		final String uploadBookmark = gestionaAPIResources.getBookmarkUrl(GESTIONA_UPLOADS_REL);

		// Creamos una nueva entrada en uploads y le subimos el documento
		final String uploadURL = uploadService.createEmptyUpload(uploadBookmark);
		final String fileName = uploadService.uploadDocument(uploadURL, documentPath);

		// Creamos un DocumentModel con el link al contenido que hemos subido y devolvemos la referencia al documento creado
		return documentService.createDocument(documentsAndFoldersURL, uploadURL, fileName);
	}

	/**
	 * Procesa el documento con el código "documentCode" y con el circuito de tramitación "circuitTemplateModel".
	 *
	 * @param documentReference
	 * 		modelo del documento a tramitar
	 * @param circuitTemplateModel
	 * 		modelo del circuito de tramitación con el que se va a tramitar el documento
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya tramitado el documento
	 */
	public void processDocument(String documentReference, CircuitTemplateModel circuitTemplateModel) throws GestionaAPIException {

		documentService.processDocument(documentReference, circuitTemplateModel);
	}

	/**
	 * Obtiene el estado de tramitación del documento con referencia en Gestiona "documentReference".
	 *
	 * @param documentReference
	 * 		referencia al documento en Gestiona
	 *
	 * @return el estado de tramitación del documento mapeado a un DocumentProcessStateModel
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se haya obtenido el estado de tramitación
	 */
	public DocumentProcessStateModel checkDocumentProcessStatus(String documentReference) throws GestionaAPIException {

		return documentService.checkDocumentProcessStatus(documentReference);
	}

	/**
	 * Saca el modelo de la anotación asociada al documento con referencia en Gestiona `documentReference`
	 *
	 * @param documentReference
	 * 		referencia al documento en Gestiona
	 *
	 * @return el modelo de la anotación
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se encuentre el documento
	 */
	public FileDocumentAnnotationModel getFileDocumentAnnotation(String documentReference) throws GestionaAPIException {

		final List<FileDocumentAnnotationModel> fileDocumentAnnotationModelList = documentService.getDocumentAnnotations(documentReference);

		return (fileDocumentAnnotationModelList == null || fileDocumentAnnotationModelList.isEmpty()) ? null : fileDocumentAnnotationModelList.getFirst();

	}

	/**
	 * Obtiene la salida del tercero a añadir al listado del circuito de tramitación a través del dni del tercero
	 *
	 * @param nif
	 * 		dni del tercero
	 *
	 * @return el modelo a añadir al listado del circuito de tramitación `thirds_output`
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no haya encontrado al tercero
	 */
	public ThirdOutputModel getThirdOutputByNif(String nif) throws GestionaAPIException {

		final ThirdModel thirdModel = thirdService.getThirdByNif(nif);

		final ThirdOutputModel thirdOutputModel = thirdService.createThirdOutputModel(thirdModel);

		if (thirdModel.getNotification_channel().equals(NotificationChannel.PAPER)) {
			final ThirdAddressModel thirdAddressModel = thirdService.getThirdAddressModel(obtainLinkByRel(thirdModel.getLinks(), DEFAULT_ADDRESS));
			thirdService.addPaperNecessaryFields(thirdOutputModel, thirdAddressModel);
		}

		return thirdOutputModel;
	}

	/**
	 * Obtiene el modelo del tercero a través de su dni y crea el modelo `ThirdPartyModel`
	 *
	 * @param nif
	 * 		dni del tercero
	 *
	 * @return el modelo `ThirdPartyModel` creado a partir de los datos del tercero
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API o que no se encuentre al tercero
	 */
	public ThirdPartyModel getThirdPartyByNif(String nif) throws GestionaAPIException {

		final ThirdModel thirdModel = thirdService.getThirdByNif(nif);

		return thirdService.createThirdPartyModel(thirdModel);
	}

	/**
	 * Añade el tercero al registro de entrada
	 *
	 * @param registryModel
	 * 		modelo del registro de entrada
	 * @param thirdPartyModel
	 * 		modelo del tercero
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API
	 */
	public void addThirdPartyToInputRegistry(RegistryAnnotationModel registryModel, ThirdPartyModel thirdPartyModel) throws GestionaAPIException {

		final String registryUrl = obtainLinkByRel(registryModel.getLinks(), THIRD_PARTIES_REL);
		thirdService.addThirdPartyModel(registryUrl, thirdPartyModel);
	}

	/**
	 * Finaliza el registro de entrada
	 *
	 * @param registryModel
	 * 		modelo del registro de entrada
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API
	 */
	public void finalizeInputRegistry(RegistryAnnotationModel registryModel) throws GestionaAPIException {
		registryService.finalizeRegistryAnnotation(registryModel);
	}

	/**
	 * Obtiene los datos del documento de Gestiona a partir de la referencia al recurso
	 *
	 * @param documentReference
	 * 		referencia al recurso del documento en Gestiona
	 *
	 * @return los datos del documento
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API
	 */
	public DocumentModel getDocument(String documentReference) throws GestionaAPIException {

		return documentService.obtainDocument(documentReference);
	}

	/**
	 * Obtiene el contenido de un documento de Gestiona
	 *
	 * @param documentModel
	 * 		datos del documento a obtener
	 *
	 * @return el contenido del documento
	 *
	 * @throws GestionaAPIException
	 * 		en caso de error en la llamada a la API
	 */
	public byte[] getDocumentContent(DocumentModel documentModel) throws GestionaAPIException {

		return documentService.getDocumentContent(documentModel);
	}

	/**
	 * Comprueba si el tercero existe en Gestiona a través de su nif, y en el caso de que no sea así, lo crea
	 *
	 * @param nif
	 * 		nif del tercero a buscar en Gestiona
	 */
	public void checkIfThirdExists(String nif) {
		try {
			final ThirdModel thirdModel = thirdService.getThirdByNif(nif);
			log.info("El tercero con dni {} y nombre {} existe en Gestiona", nif, thirdModel.getFull_name());

		} catch (GestionaAPIException e) {
			log.info("Se va a crear el tercero con DNI {} en Gestiona", nif);
			thirdService.createThird(nif);
		}
	}

}
