package main.gestiona.apirest;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.CommandLineRunner;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import main.exception.GestionaAPIException;
import main.gestiona.apirest.config.GestionaApiConfig;
import main.gestiona.apirest.model.BookmarkModel;
import main.gestiona.apirest.model.LinkModel;
import main.gestiona.apirest.model.PageModel;

@Service
public class GestionaAPIResources implements CommandLineRunner {
	public static final String CONTENT_TYPE = "Content-Type";
	public static final String X_GESTIONA_ACCESS_TOKEN = "X-Gestiona-Access-Token";
	public static final String DOCUMENTS_AND_FOLDERS_REL = "documents-and-folders";
	public static final String SELF_REL = "self";
	public static final String NEXT_PAGE_REL = "next";
	private static final Logger log = LoggerFactory.getLogger(GestionaAPIResources.class);
	private static final String FORMAT_BOOKMARKS = "%s/rest";

	/**
	 * Obtiene el link definido por "rel" del listado de links
	 *
	 * @param linkModelList
	 * 		listado de links de un objeto
	 * @param rel
	 * 		relación del link a buscar
	 *
	 * @return la url del link buscado
	 *
	 * @throws GestionaAPIException
	 * 		si no se ha encontrado el link con la relación especificada
	 */
	public static String obtainLinkByRel(List<LinkModel> linkModelList, String rel) throws GestionaAPIException {

		return linkModelList.stream()//
				.filter(link -> rel.equalsIgnoreCase(link.getRel()))//
				.findFirst()//
				.orElseThrow(() -> new GestionaAPIException("No se ha encontrado un link con la relación: " + rel))//
				.getHref();
	}

	private final RestTemplate restTemplate;
	private final String gestionaApiUrl;
	private final String accessToken;
	private List<LinkModel> bookmarks;

	public GestionaAPIResources(GestionaApiConfig gestionaApiConfig) {
		this.restTemplate = new RestTemplate();
		this.gestionaApiUrl = gestionaApiConfig.getUrl();
		this.accessToken = gestionaApiConfig.getAccessToken();
	}

	@Override
	public void run(String... args) {
		try {
			bookmarks = obtainBookmarks();

		} catch (GestionaAPIException e) {
			bookmarks = Collections.emptyList();
			log.error(e.getMessage());
		}
	}

	/**
	 * Obtiene la url del bookmark a través de su relación
	 *
	 * @param linkRel
	 * 		relación del bookmark a buscar
	 *
	 * @return la url del bookmark buscado
	 *
	 * @throws GestionaAPIException
	 * 		en caso de no encontrar el bookmark con la relación especificada
	 */
	public String getBookmarkUrl(String linkRel) throws GestionaAPIException {
		reloadBookmarksIfNotPresent();
		return bookmarks.stream()//
				.filter(bookmark -> linkRel.equalsIgnoreCase(bookmark.getRel()))//
				.findFirst()//
				.orElseThrow(() -> new GestionaAPIException(String.format("No se ha encontrado un bookmark con la relación: %s", linkRel)))//
				.getHref();
	}

	/**
	 * Función para crear una petición a la API de Gestiona genérica a través de las especificaciones enviadas a través de sus parámetros
	 *
	 * @param uri
	 * 		url sobre la cual hacer la petición
	 * @param method
	 * 		método de la petición
	 * @param extraHeaders
	 * 		cabeceras extras, aparte del token de acceso de Gestiona
	 * @param body
	 * 		cuerpo de la petición, en el caso de que se requiera
	 * @param responseType
	 * 		tipo de dato sobre el cual nos va a llegar la respuesta
	 * @param <T>
	 * 		tipo de dato con el que se va a trabajar
	 *
	 * @return la respuesta obtenida de la API tras hacer la petición HTTP especificada
	 */
	public <T> ResponseEntity<T> createPetition(String uri, HttpMethod method, Map<String, String> extraHeaders, Object body, Class<T> responseType)
			throws RestClientException {

		final HttpHeaders headers = new HttpHeaders();
		headers.set(X_GESTIONA_ACCESS_TOKEN, accessToken);

		if (extraHeaders != null && !extraHeaders.isEmpty()) {
			headers.setAll(extraHeaders);
		}

		final HttpEntity<?> entity = (body != null) ? new HttpEntity<>(body, headers) : new HttpEntity<>(headers);

		return restTemplate.exchange(uri, method, entity, responseType);
	}

	/**
	 * Obtiene la siguiente página de un modelo genérico de página
	 *
	 * @param pageModel
	 * 		modelo de la página actual
	 * @param pageModelClass
	 * 		tipo de dato de la página
	 * @param <T>
	 * 		tipo de dato de la página
	 *
	 * @return el modelo de la siguiente página
	 *
	 * @throws GestionaAPIException
	 * 		en caso de que no se pueda obtener la siguiente página
	 */
	public <T> PageModel<T> getNextPage(PageModel<T> pageModel, Class<? extends PageModel<T>> pageModelClass) throws GestionaAPIException {

		final String nextPageURL = obtainLinkByRel(pageModel.getLinks(), NEXT_PAGE_REL);

		if (nextPageURL == null) {
			return null;
		}

		return createPetition(nextPageURL, HttpMethod.GET, null, null, pageModelClass).getBody();
	}

	/**
	 * Obtiene los bookmarks de la API a través de la petición GET a la ruta "/rest"
	 *
	 * @return el listado de LinkModels con los bookmarks que ha obtenido
	 *
	 * @throws GestionaAPIException
	 * 		en el caso de que no se hayan podido leer los bookmarks correctamente
	 */
	private List<LinkModel> obtainBookmarks() throws GestionaAPIException {
		try {

			final String url = String.format(FORMAT_BOOKMARKS, gestionaApiUrl);

			return Objects.requireNonNull(createPetition(url, HttpMethod.GET, null, null, BookmarkModel.class).getBody()).getLinks();

		} catch (RestClientException e) {
			throw new GestionaAPIException("No se han podido leer los bookmarks correctamente" + e.getMessage());
		}
	}

	/**
	 * Si los bookmarks no se han cargado al lanzar la aplicación los vuelve a cargar
	 *
	 * @throws GestionaAPIException
	 * 		si se sigue sin poder acceder a los bookmarks
	 */
	private void reloadBookmarksIfNotPresent() throws GestionaAPIException {
		if (bookmarks.isEmpty()) {
			bookmarks = obtainBookmarks();
		}
	}

	public enum NotificationChannel {
		TELEMATIC,
		PAPER
	}

	public enum RelationType {
		INVOLVED,
		REPRESENTANT,
		SOLICITOR,
		COMPLAINANT,
		DENOUNCED,
		TENDERER,
		OWNER,
		TRANSFERER,
		AQUIRER,
		RECIPIENT,
		APPLICANT,
		PROVIDER,
		REGISTRY,
		CONTRACTING,
		ANY_SOLICITOR,
		SUPPLIER
	}

}
