package main.controller;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import main.connectors.thirds.view.service.GoogleSheetsService;
import main.gestiona.apirest.model.CircuitTemplateModel;
import main.model.DocumentaryFlowModel;
import main.repository.SupportTicketService;
import main.service.AcademyService;
import main.service.ApiDocService;
import main.service.DataSetService;
import main.service.DocumentaryFlowService;
import main.service.GesCodeService;
import main.service.GithubProjectService;
import main.service.IncidenceService;

@Configuration
@Controller
public class MockupController {
	private static final String SUPPORT_PAGE = "support";
	private static final String INCIDENCE_PAGE = "incidences";
	private static final String DOCUMENTARY_FLOW_PAGE = "documentary-flow";
	private static final String HOME_PAGE = "home";
	private static final String ACADEMY_PAGE = "academy";
	private static final String API_DOC_PAGE = "api-doc";
	private static final String DATA_SET_PAGE = "data-set";
	private static final String REDIRECT_PAGE_FORMAT = "redirect:%s";
	private static final String IFRAME_TEMPLATE = "iframe";
	private static final String MAIN_LAYOUT_TEMPLATE = "main-layout";
	private static final String MOCKUPS_TEMPLATE = "mockups";
	private static final String MOCKUP_MENU_OPTION_ATTRIBUTE = "mockupMenuOption";
	private static final String GIT_HUB_URL_ATTRIBUTE = "githubProjectURL";
	private static final String GES_CODE_URL_ATTRIBUTE = "gesCodeIframeURL";
	private static final String IFRAME_URL_ATTRIBUTE = "iframeURL";
	private static final String PAGE_ATTRIBUTE = "page";
	private static final String TEMPLATE_ATTRIBUTE = "template";
	private static final String INCIDENCE_LIST_ATTRIBUTE = "incidenceList";
	private static final String DOCUMENTARY_FLOW_LIST_ATTRIBUTE = "documentaryFlowList";
	private static final String MESSAGE_ATTRIBUTE = "message";
	private static final String UPLOAD_DOCUMENT_PAGE = "/upload-document";
	private static final String PROCESS_DOCUMENT_PAGE = "/process-document";
	private static final Logger log = LoggerFactory.getLogger(MockupController.class);
	private static final String DOCUMENTARY_FLOW_ATTRIBUTE = "documentaryFlow";
	private static final String REGISTER_INCIDENCE_PAGE = "register-incidence";
	private final ObjectMapper objectMapper;
	@Autowired
	private DocumentaryFlowService documentaryFlowService;

	@Autowired
	@Qualifier("supportTicketService")
	private SupportTicketService supportTicketService;

	@Autowired
	private IncidenceService incidenceService;

	@Autowired
	private GoogleSheetsService googleSheetsService;

	@Autowired
	private AcademyService academyService;

	@Autowired
	private ApiDocService apiDocService;

	@Autowired
	private DataSetService dataSetService;

	@Autowired
	private GesCodeService gesCodeService;

	@Autowired
	private GithubProjectService githubProjectService;

	public MockupController() {
		this.objectMapper = new ObjectMapper();
	}

	/**
	 * Setea el valor de los atributos globales al modelo que le pasaremos a cada vista
	 *
	 * @param model
	 * 		modelo en el que se da valor a los atributos
	 */
	@ModelAttribute
	public void addAttributes(Model model) {
		model.addAttribute(GIT_HUB_URL_ATTRIBUTE, githubProjectService.getGithubProjectUrl());
		model.addAttribute(GES_CODE_URL_ATTRIBUTE, gesCodeService.getGesCodeIframeUrl());
	}

	/**
	 * Mapea el path "/" a la plantilla principal y setea los atributos necesarios para mostrar la página de inicio
	 *
	 * @param model
	 * 		modelo de datos al que pasaremos el atributo academyIframeURL
	 *
	 * @return el nombre de la plantilla principal
	 */
	@GetMapping("/")
	public String home(Model model) {
		return loadMainLayoutTemplate(model, null, HOME_PAGE);
	}

	/**
	 * Mapea el path "/api-doc" a la plantilla principal y setea los atributos necesarios para mostrar la página api-doc
	 *
	 * @param model
	 * 		modelo de datos al que pasaremos el atributo academyIframeURL
	 *
	 * @return el nombre de la plantilla principal
	 */
	@GetMapping("/api-doc")
	public String apiDocPage(Model model) {
		return loadIframeTemplate(model, API_DOC_PAGE, apiDocService.getApiDocIframeUrl());
	}

	/**
	 * Mapea el path "/academy" a la plantilla principal y setea los atributos necesarios para mostrar la página academy
	 *
	 * @param model
	 * 		modelo de datos al que pasaremos el atributo academyIframeURL
	 *
	 * @return el nombre de la plantilla principal
	 */
	@GetMapping("/academy")
	public String academyPage(Model model) {
		return loadIframeTemplate(model, ACADEMY_PAGE, academyService.getAcademyIframeUrl());
	}

	/**
	 * Mapea el path "/data-set" a la plantilla principal y setea los atributos necesarios data-set
	 *
	 * @param model
	 * 		modelo de datos al que pasaremos el atributo academyIframeURL
	 *
	 * @return el nombre de la plantilla principal
	 */
	@GetMapping("/data-set")
	public String dataSetPage(Model model) {
		return loadIframeTemplate(model, DATA_SET_PAGE, dataSetService.getDataSetIframeUrl());
	}

	/**
	 * Mapea el path "/documentary-flow" a la plantilla principal, sincroniza la base de datos con los expedientes de Gestiona, actualiza los estados de los
	 * flujos documentales en función de los estados de la tramitación de sus documentos y setea los atributos necesarios para mostrar la página
	 * documentary-flow
	 *
	 * @param model
	 * 		modelo de datos al que pasaremos el atributo academyIframeURL
	 *
	 * @return el nombre de la plantilla principal
	 */
	@GetMapping("/documentary-flow")
	public String documentaryFlowPage(Model model) {

		documentaryFlowService.documentaryFlowDatabaseSync();
		documentaryFlowService.updateProcessStatus();

		model.addAttribute(DOCUMENTARY_FLOW_LIST_ATTRIBUTE, documentaryFlowService.getAll());

		return loadMockupTemplate(model, DOCUMENTARY_FLOW_PAGE);
	}

	/**
	 * Mapea el path "/support" a la plantilla principal y setea el valor de los atributos necesarios para mostrar la página support
	 *
	 * @param model
	 * 		modelo de datos donde se setea el valor de los atributos
	 *
	 * @return nombre de plantilla principal
	 */
	@GetMapping("/support")
	public String getSoporte(Model model) {
		return loadMainLayoutTemplate(model, SUPPORT_PAGE, SUPPORT_PAGE);
	}

	/**
	 * Mapea la url "/incidences a la vista "incidences.html" y le pasa por parámetro la url a la página de las incidencias"
	 *
	 * @param model
	 * 		modelo de datos al que pasamos el valor de los atributos de la vista
	 *
	 * @return nombre de la vista "incidences" a la que se le ha mapeado "/incidences"
	 */
	@GetMapping("/incidences")
	public String incidencesPage(Model model) {
		model.addAttribute(INCIDENCE_LIST_ATTRIBUTE, incidenceService.getAll());
		return loadMockupTemplate(model, INCIDENCE_PAGE);
	}

	/**
	 * Mapea la url "/upload-document a la vista "upload-document.html" y le pasa por parámetro el modelo del flujo documental
	 *
	 * @param documentaryFlowId
	 * 		id del flujo documental
	 * @param model
	 * 		modelo de datos al que pasamos el valor de los atributos de la vista
	 *
	 * @return nombre de la vista "upload-document" a la que se le ha mapeado "/upload-document"
	 */
	@GetMapping("/upload-document")
	public String uploadDocumentPage(
			@RequestParam("id")
			String documentaryFlowId, Model model) {

		final DocumentaryFlowModel documentaryFlow = documentaryFlowService.getById(documentaryFlowId);

		model.addAttribute(DOCUMENTARY_FLOW_ATTRIBUTE, documentaryFlow);

		return loadMockupTemplate(model, UPLOAD_DOCUMENT_PAGE);
	}

	/**
	 * Mapea la url "/process-document a la vista "process-document.html" y le pasa por parámetro el listado de circuitos de tramitación y la referencia al
	 * documento
	 *
	 * @param documentReference
	 * 		referencia al recurso del documento en Gestiona
	 * @param model
	 * 		modelo de datos al que pasamos el valor de los atributos de la vista
	 *
	 * @return nombre de la vista "process-document" a la que se le ha mapeado "/process-document"
	 */
	@GetMapping("/process-document")
	public String processDocumentPage(
			@RequestParam("documentaryFlowId")
			String documentaryFlowId,
			@RequestParam("document")
			String documentReference, Model model) {

		final List<CircuitTemplateModel> circuitTemplateModelList = documentaryFlowService.getCircuitTemplates(documentReference);
		model.addAttribute("circuitTemplates", circuitTemplateModelList);
		model.addAttribute("documentReference", documentReference);
		model.addAttribute("documentaryFlowId", documentaryFlowId);
		model.addAttribute("thirdsList", googleSheetsService.getAllThirds());

		return loadMockupTemplate(model, PROCESS_DOCUMENT_PAGE);
	}

	/**
	 * Mapea la url "/register-incidence" a la vista "register-incidence.html"
	 *
	 * @param model
	 * 		modelo de datos al que pasamos el valor de los atributos de la vista
	 *
	 * @return nombre de la vista "register-incidence" a la que se le ha mapeado "/register-incidence"
	 */
	@GetMapping("/register-incidence")
	public String registerIncidencePage(Model model) {

		model.addAttribute("thirdsList", googleSheetsService.getAllThirds());

		return loadMockupTemplate(model, REGISTER_INCIDENCE_PAGE);
	}

	/**
	 * Maneja el registro de una nueva incidencia
	 *
	 * @param nif
	 * 		dni del tercero a asignarle la incidencia
	 * @param subject
	 * 		asunto de la incidencia
	 *
	 * @return una redirección a la página de las incidencias
	 */
	@PostMapping("/register-incidence")
	public String registerNewIncidence(
			@RequestParam("nif")
			String nif,
			@RequestParam("subject")
			String subject) {

		incidenceService.registerNewIncidence(nif, subject);

		return String.format(REDIRECT_PAGE_FORMAT, INCIDENCE_PAGE);
	}

	/**
	 * Maneja la carga de un documento asociado a un flujo documental.
	 *
	 * @param documentaryFlowId
	 * 		el ID del flujo documental al que se asocia el documento
	 * @param file
	 * 		el archivo del documento cargado en forma de MultipartFile
	 *
	 * @return una redirección a la página del flujo documental
	 */
	@PostMapping("/upload-document")
	public String handleDocumentUpload(
			@RequestParam("documentaryFlowId")
			String documentaryFlowId,
			@RequestParam("fileIdGestiona")
			String fileIdGestiona,
			@RequestParam("fileInput")
			MultipartFile file) {

		documentaryFlowService.handleDocumentUpload(documentaryFlowId, fileIdGestiona, file);

		return String.format(REDIRECT_PAGE_FORMAT, DOCUMENTARY_FLOW_PAGE);
	}

	/**
	 * Maneja la tramitación de un documento con un circuito seleccionado
	 *
	 * @param documentReference
	 * 		referencia al documento en Gestiona
	 * @param circuitJson
	 * 		circuito de tramitación en JSON
	 * @param nif
	 * 		dni del tercero a notificar, si procede
	 *
	 * @return una redirección a la página del flujo documental
	 */
	@PostMapping("/process-document")
	public String processDocument(
			@RequestParam("documentaryFlowId")
			String documentaryFlowId,
			@RequestParam("documentReference")
			String documentReference,
			@RequestParam("circuit")
			String circuitJson,
			@RequestParam("dni")
			String nif) {

		try {

			final CircuitTemplateModel circuit = objectMapper.readValue(circuitJson, CircuitTemplateModel.class);
			documentaryFlowService.processDocument(documentaryFlowId, documentReference, circuit, nif);

		} catch (JsonProcessingException exception) {
			log.error("Ha ocurrido un error en la tramitación del documento");
		}

		return String.format(REDIRECT_PAGE_FORMAT, DOCUMENTARY_FLOW_PAGE);
	}

	/**
	 * Maneja la apertura de un ticket de soporte
	 *
	 * @param ticketTitle
	 * 		asunto del ticket
	 * @param redirectAttributes
	 * 		modelo de datos con donde se setea el atributo msg
	 *
	 * @return una redirección a la página de soporte
	 */
	@PostMapping("/open-ticket")
	public String handleTicketOpening(
			@RequestParam("ticketTitle")
			String ticketTitle, RedirectAttributes redirectAttributes) {

		final String message = supportTicketService.handleTicketOpening(ticketTitle);
		redirectAttributes.addFlashAttribute(MESSAGE_ATTRIBUTE, message);

		return String.format(REDIRECT_PAGE_FORMAT, SUPPORT_PAGE);
	}

	/**
	 * Maneja la descarga del pdf con el documento tramitado
	 *
	 * @param documentReference
	 * 		referencia al documento que ha sido tramitado
	 *
	 * @return la respuesta con el documento a descargar
	 */
	@GetMapping("/download-pdf")
	public ResponseEntity<byte[]> downloadPDF(
			@RequestParam("documentReference")
			String documentReference) {

		return documentaryFlowService.downloadProcessedDocument(documentReference);
	}

	/**
	 * Método que setea a un modelo la opcíon del menú que se mostrará en la página mockups y la plantilla que seguirá
	 *
	 * @param model
	 * 		modelo de datos donde se setean los atributos
	 *
	 * @return el nombre de la plantilla principal
	 */
	private String loadMockupTemplate(Model model, String menuOption) {
		model.addAttribute(MOCKUP_MENU_OPTION_ATTRIBUTE, menuOption);
		return loadMainLayoutTemplate(model, MOCKUPS_TEMPLATE, MOCKUPS_TEMPLATE);
	}

	/**
	 * Método que setea a un modelo la url y página pasada por parámetro y la plantilla iframe que seguirá la página
	 *
	 * @param model
	 * 		modelo de datos donde se setean los atributos
	 * @param page
	 * 		nombre de la página que se mostrará
	 * @param iframeUrl
	 * 		url del iframe que se mostrará embebido en la página
	 *
	 * @return el nombre de la plantilla principal
	 */
	private String loadIframeTemplate(Model model, String page, String iframeUrl) {
		model.addAttribute(IFRAME_URL_ATTRIBUTE, iframeUrl);
		return loadMainLayoutTemplate(model, page, IFRAME_TEMPLATE);
	}

	/**
	 * Método que setea a un modelo la página y plantilla pasados por parámetro, luego devuelve el nombre de la plantilla principal
	 *
	 * @param model
	 * 		modelo de datos donde se setean los atributos
	 * @param page
	 * 		nombre de la página que se mostrará
	 * @param template
	 * 		plantilla de la página
	 *
	 * @return nombre de la plantilla principal
	 */
	private String loadMainLayoutTemplate(Model model, String page, String template) {
		model.addAttribute(PAGE_ATTRIBUTE, page);
		model.addAttribute(TEMPLATE_ATTRIBUTE, template);
		return MAIN_LAYOUT_TEMPLATE;
	}
}
