package main.connectors.thirds.view.service;

import java.io.FileInputStream;
import java.io.IOException;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.google.api.client.googleapis.auth.oauth2.GoogleCredential;
import com.google.api.services.sheets.v4.Sheets;

import main.connectors.thirds.view.config.GoogleSheetsConfig;

@Service
public class GoogleSheetsService {
	private static final String APPLICATION_NAME = "MockupForDevelopers";
	private static final int NIF_COLUMN = 1;
	private static final int INCIDENCE_NUMBER_COLUMN = 0;
	private static final int INCIDENCE_GROUP_COLUMN = 11;
	private static final Logger log = LoggerFactory.getLogger(GoogleSheetsService.class);
	private static final String range = "Persona!A2:L";

	private final String credentialsPath;
	private final String spreadsheetId;
	private final SearchType searchType;

	public GoogleSheetsService(GoogleSheetsConfig googleSheetsConfig) {
		this.credentialsPath = googleSheetsConfig.getCredentialsPath();
		this.spreadsheetId = googleSheetsConfig.getSpreadsheetId();
		this.searchType = googleSheetsConfig.getSearchType();
	}

	/**
	 * Obtiene el servicio de hojas de cálculo de google
	 *
	 * @return el servicio
	 *
	 * @throws IOException
	 * 		en el caso de que ocurra un error al leer el fichero con las credenciales
	 */
	public Sheets getSheetsService() throws IOException {

		final GoogleCredential credential = GoogleCredential.fromStream(new FileInputStream(credentialsPath)).createScoped(
				List.of("https://www.googleapis.com/auth/spreadsheets"));

		return new Sheets.Builder(credential.getTransport(), credential.getJsonFactory(), credential)//
				.setApplicationName(APPLICATION_NAME)//
				.build();

	}

	/**
	 * Busca en la hoja el dni del tercero y en caso de que lo encuentre, devuelve su fila. En caso contrario devuelve null
	 *
	 * @param nif
	 * 		dni del tercero
	 *
	 * @return una lista con los valores del tercero de la tabla o null en el caso de que no lo encuentre
	 */
	public List<Object> getThirdByNIF(String nif) {

		final List<List<Object>> values = getAllThirds();

		return resolveThirdSearch(nif, values, NIF_COLUMN);

	}

	/**
	 * Busca al tercero por el campo indicado a nivel de configuración
	 *
	 * @param searchedValue
	 * 		valor del tercero buscado
	 *
	 * @return una lista con los valores del tercero de la tabla o null en el caso de que no lo encuentre
	 */
	public List<Object> resolveThirdSearch(String searchedValue) {
		final List<List<Object>> values = getAllThirds();

		return switch (searchType) {
			case SearchType.NIF -> resolveThirdSearch(searchedValue, values, NIF_COLUMN);
			case SearchType.INCIDENCE_GROUP -> resolveThirdSearch(searchedValue, values, INCIDENCE_GROUP_COLUMN);
			case SearchType.INCIDENCE_NUMBER -> resolveThirdSearch(searchedValue, values, INCIDENCE_NUMBER_COLUMN);
			default -> {
				log.error("Tipo de búsqueda desconocido: {}", searchType);
				yield null;
			}
		};
	}

	/**
	 * Obtiene todos los registros en la hoja de Google Sheets con los terceros
	 *
	 * @return un listado de listas de objetos siendo cada una de estas los registros para cada uno de los terceros
	 */
	public List<List<Object>> getAllThirds() {
		try {
			final Sheets service = getSheetsService();

			return service.spreadsheets().values()//
					.get(spreadsheetId, range)//
					.execute().getValues();

		} catch (IOException e) {
			log.error("Error al obtener los terceros de Google Sheets: {}", e.getMessage());
			return null;
		}
	}

	/**
	 * Busca al tercero a través de un valor `searchedValue` de la columna `column`
	 *
	 * @param searchedValue
	 * 		el valor del tercero a buscar
	 * @param values
	 * 		los valores obtenidos de la hoja de Google
	 * @param column
	 * 		la columna en la que se encuentra el valor buscado
	 *
	 * @return una lista con los valores del tercero de la tabla o null en el caso de que no lo encuentre
	 */
	private List<Object> resolveThirdSearch(String searchedValue, List<List<Object>> values, int column) {

		return values.stream()//
				.filter((List<Object> row) -> row.size() > column && row.get(column).equals(searchedValue))//
				.findFirst()//
				.orElse(null);//
	}

	public enum SearchType {
		INCIDENCE_NUMBER,
		INCIDENCE_GROUP,
		NIF
	}

}