package main.connectors.thirds.view.controller;

import static main.connectors.common.service.ConnectorUtilsService.CAUSE_NOT_FOUND;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_0;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_1;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_2;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_3;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_4;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_5;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_6;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_7;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_8;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_9;
import static main.connectors.common.service.ConnectorUtilsService.MEDIA_TYPE_GENERIC_OP;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;

import jakarta.servlet.http.HttpServletRequest;
import main.connectors.common.exception.SignatureErrorException;
import main.connectors.common.model.DataModel;
import main.connectors.common.model.ErrorModel;
import main.connectors.common.model.FieldModel;
import main.connectors.common.response.BookmarkResponse;
import main.connectors.common.response.VersionResponse;
import main.connectors.common.service.ConnectorUtilsService;
import main.connectors.thirds.view.service.GoogleSheetsService;

@Controller
@RequestMapping(ThirdConnectorController.CONNECTOR_URL_PREFIX)
public class ThirdConnectorController {

	public static final String CONNECTOR_URL_PREFIX = "/connectors/third-data";
	private static final int GROUP_COLUMN = 11;
	private static final int CLASS_COLUMN = 10;
	private static final int CATEGORY_COLUMN = 9;
	private static final int SUBSCRIPTED_COLUMN = 8;
	private static final int DOMICILIATION_COLUMN = 7;
	private static final int DEBTS_COLUMN = 6;
	private static final int RESIDENT_COLUMN = 5;
	private static final int BIRTH_COLUMN = 4;
	private static final int NAME_COLUMN = 2;
	private static final int NIF_COLUMN = 1;
	private static final String GENERIC_OPERATION_FORMAT = "%s%s/genericoperations";
	private static final String MESSAGE_NOT_FOUND = "Third not found";
	@Autowired
	private ConnectorUtilsService connectorUtilsService;

	@Autowired
	private GoogleSheetsService googleSheetsService;

	/**
	 * Obtiene la versión del conector de búsqueda de terceros
	 *
	 * @param token
	 * 		token de autenticación
	 *
	 * @return ResponseEntity con la versión
	 *
	 * @throws SignatureErrorException
	 * 		si no se ha podido calcular la firma
	 */
	@GetMapping("/versions/current")
	public ResponseEntity<VersionResponse> getCurrentVersion(
			@RequestHeader("X-Rest-Basic-Token")
			String token) throws SignatureErrorException {

		// TODO: habrá que validar el token

		return connectorUtilsService.generateVersionResponse(token);
	}

	/**
	 * Obtiene la lista de bookmarks del conector de búsqueda de terceros
	 *
	 * @param token
	 * 		token de autenticación
	 * @param request
	 * 		request con información sobre la petición HTTP
	 *
	 * @return ResponseEntity con la lista de bookmarks o un status de Unauthorized si la versión no coincide con la versión del conector de búsqueda de
	 * terceros
	 *
	 * @throws SignatureErrorException
	 * 		si no se ha podido calcular la firma
	 */
	@GetMapping("/")
	public ResponseEntity<BookmarkResponse> getBookmarkList(
			@RequestHeader("X-Rest-Basic-Token")
			String token, HttpServletRequest request) throws SignatureErrorException {

		// TODO: habrá que validar el token

		final String genericOperationLink = String.format(GENERIC_OPERATION_FORMAT, connectorUtilsService.obtainServerUrl(request), CONNECTOR_URL_PREFIX);

		return connectorUtilsService.generateBookmarkResponse(genericOperationLink, token);
	}

	/**
	 * Obtiene el identificador de la incidencia para un número de registro y número de archivo específicos
	 *
	 * @param token
	 * 		token de autenticación
	 * @param requestData
	 * 		datos de la petición
	 *
	 * @return ResponseEntity con el resultado de la operación, o un status de PRECONDITION_FAILED si no encuentra la incidencia, o un status de Unauthorized si
	 * la versión no coincide con la versión del conector de incidencias
	 *
	 * @throws SignatureErrorException
	 * 		si no se ha podido calcular la firma
	 */
	@PostMapping("/genericoperations")
	public ResponseEntity<?> getGenericOperation(
			@RequestHeader("X-Rest-Basic-Token")
			String token,
			@RequestBody
			DataModel requestData) throws SignatureErrorException {

		// TODO: habrá que validar el token

		final Map<String, FieldModel> data = requestData.getData();

		final Set<String> expectedFields = Set.of(FIELD_0);
		final ResponseEntity<ErrorModel> errorResponse = connectorUtilsService.checkErrors(data, expectedFields);

		// En el caso de que haya errores, devolvemos la respuesta con el error
		if (errorResponse != null) {
			return errorResponse;
		}

		// Si no hay errores con los parámetros, tratamos de resolver la operación genérica
		return resolveGenericOperation(data, token);
	}

	/**
	 * Resuelve la operación genérica. Si encuentra al tercero, devuelve un status de OK con el resultado de la operación. Si no la encuentra, devuelve un error
	 * informando de que el elemento buscado no existe
	 *
	 * @param data
	 * 		datos de la petición
	 * @param token
	 * 		token de autenticación
	 *
	 * @return resultado de la operación
	 */
	private ResponseEntity<?> resolveGenericOperation(Map<String, FieldModel> data, String token) {
		final String searchedValue = data.get(FIELD_0).getValue();

		final List<Object> thirdData = googleSheetsService.resolveThirdSearch(searchedValue);

		// Si no encuentra al tercero, devolvemos un error indicando que no existe
		if (thirdData == null) {
			return connectorUtilsService.buildErrorResponse(MESSAGE_NOT_FOUND, CAUSE_NOT_FOUND, null);
		}

		final HttpHeaders headers = connectorUtilsService.createHeaders(MEDIA_TYPE_GENERIC_OP, token);

		// Construimos la respuesta a la petición
		final Map<String, FieldModel> responseMap = buildResponseMap(thirdData);
		final DataModel responseModel = new DataModel(responseMap);

		return new ResponseEntity<>(responseModel, headers, HttpStatus.OK);
	}

	/**
	 * Crea el mapa de respuesta con los datos del tercero
	 *
	 * @param thirdData
	 * 		los datos del tercero
	 *
	 * @return el mapa a devolver en la respuesta
	 */
	private Map<String, FieldModel> buildResponseMap(List<Object> thirdData) {
		final Map<String, FieldModel> map = new HashMap<>();

		map.put(FIELD_0, new FieldModel(thirdData.get(NIF_COLUMN).toString()));
		map.put(FIELD_1, new FieldModel(thirdData.get(NAME_COLUMN).toString()));
		map.put(FIELD_2, new FieldModel(thirdData.get(BIRTH_COLUMN).toString()));
		map.put(FIELD_3, new FieldModel(thirdData.get(RESIDENT_COLUMN).toString()));
		map.put(FIELD_4, new FieldModel(thirdData.get(DEBTS_COLUMN).toString()));
		map.put(FIELD_5, new FieldModel(thirdData.get(DOMICILIATION_COLUMN).toString()));
		map.put(FIELD_6, new FieldModel(thirdData.get(SUBSCRIPTED_COLUMN).toString()));
		map.put(FIELD_7, new FieldModel(thirdData.get(CATEGORY_COLUMN).toString()));
		map.put(FIELD_8, new FieldModel(thirdData.get(CLASS_COLUMN).toString()));
		map.put(FIELD_9, new FieldModel(thirdData.get(GROUP_COLUMN).toString()));

		return map;
	}

}
