package main.connectors.incidences.view.controller;

import static main.connectors.common.service.ConnectorUtilsService.CAUSE_NOT_FOUND;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_0;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_1;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_2;
import static main.connectors.common.service.ConnectorUtilsService.FIELD_3;
import static main.connectors.common.service.ConnectorUtilsService.INCIDENCE_MESSAGE_NOT_FOUND;
import static main.connectors.common.service.ConnectorUtilsService.MEDIA_TYPE_GENERIC_OP;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import jakarta.servlet.http.HttpServletRequest;
import main.connectors.common.exception.SignatureErrorException;
import main.connectors.common.model.DataModel;
import main.connectors.common.model.ErrorModel;
import main.connectors.common.model.FieldModel;
import main.connectors.common.response.BookmarkResponse;
import main.connectors.common.response.VersionResponse;
import main.connectors.common.service.ConnectorUtilsService;
import main.exception.IncidenceNotFoundException;
import main.model.IncidenceModel;
import main.service.IncidenceService;

@RestController
@RequestMapping(value = IncidenceDataConnectorController.INCIDENCE_DATA_CONNECTOR_URL_PREFIX)
public class IncidenceDataConnectorController {

	public static final String INCIDENCE_DATA_CONNECTOR_URL_PREFIX = "/connectors/incidence-data";
	@Autowired
	private ConnectorUtilsService connectorUtilsService;

	@Autowired
	private IncidenceService incidenceService;

	/**
	 * Obtiene la versión del conector de incidencias
	 *
	 * @param token
	 * 		token de autenticación
	 *
	 * @return ResponseEntity con la versión
	 *
	 * @throws SignatureErrorException
	 * 		si no se ha podido calcular la firma
	 */
	@GetMapping("/versions/current")
	public ResponseEntity<VersionResponse> getCurrentVersion(
			@RequestHeader("X-Rest-Basic-Token")
			String token) throws SignatureErrorException {

		// TODO: habrá que validar el token

		return connectorUtilsService.generateVersionResponse(token);
	}

	/**
	 * Obtiene la lista de bookmarks para una versión específica
	 *
	 * @param token
	 * 		token de autenticación
	 * @param request
	 * 		request con información sobre la petición HTTP
	 *
	 * @return ResponseEntity con la lista de bookmarks o un status de Unauthorized si la versión no coincide con la versión del conector de incidencias
	 *
	 * @throws SignatureErrorException
	 * 		si no se ha podido calcular la firma
	 */
	@GetMapping("/")
	public ResponseEntity<BookmarkResponse> getBookmarkList(
			@RequestHeader("X-Rest-Basic-Token")
			String token, HttpServletRequest request) throws SignatureErrorException {

		// TODO: habrá que validar el token

		final String genericOperationLink = String.format("%s%s/genericoperations", connectorUtilsService.obtainServerUrl(request),
				INCIDENCE_DATA_CONNECTOR_URL_PREFIX);

		return connectorUtilsService.generateBookmarkResponse(genericOperationLink, token);
	}

	/**
	 * Obtiene el identificador de la incidencia para un número de expediente específico
	 *
	 * @param token
	 * 		token de autenticación
	 * @param requestData
	 * 		datos de la petición
	 *
	 * @return ResponseEntity con el resultado de la operación, o un status de PRECONDITION_FAILED si no encuentra la incidencia, o un status de Unauthorized si
	 * la versión no coincide con la versión del conector de incidencias
	 *
	 * @throws SignatureErrorException
	 * 		si no se ha podido calcular la firma
	 */
	@PostMapping("/genericoperations")
	public ResponseEntity<?> getGenericOperation(
			@RequestHeader("X-Rest-Basic-Token")
			String token,
			@RequestBody
			DataModel requestData) throws SignatureErrorException {

		// TODO: habrá que validar el token

		final Map<String, FieldModel> data = requestData.getData();

		final Set<String> expectedFields = Set.of(FIELD_0);
		final ResponseEntity<ErrorModel> errorResponse = connectorUtilsService.checkErrors(data, expectedFields);

		// En el caso de que haya errores, generamos y devolvemos el error
		if (errorResponse != null) {
			return errorResponse;
		}

		// Si no hay errores con los parámetros, tratamos de resolver la operación genérica
		return resolveGenericOperation(data, token);
	}

	/**
	 * Resuelve la operación genérica. Si encuentra la incidencia, devuelve un status de OK con el resultado de la operación. Si no la encuentra, devuelve un
	 * error informando de que el elemento buscado no existe
	 *
	 * @param data
	 * 		datos de la petición
	 * @param token
	 * 		token de autenticación
	 *
	 * @return resultado de la operación
	 */
	private ResponseEntity<?> resolveGenericOperation(Map<String, FieldModel> data, String token) {

		final String fileNumber = data.get(FIELD_0).getValue();

		try {
			final IncidenceModel incidenceModel = incidenceService.searchIncidence(fileNumber);

			final HttpHeaders headers = connectorUtilsService.createHeaders(MEDIA_TYPE_GENERIC_OP, token);

			final Map<String, FieldModel> responseMap = buildResponseMap(incidenceModel);
			final DataModel responseModel = new DataModel(responseMap);

			return new ResponseEntity<>(responseModel, headers, HttpStatus.OK);

		} catch (IncidenceNotFoundException e) {
			// Si la incidencia no existe, devolvemos un error 412
			return connectorUtilsService.buildErrorResponse(INCIDENCE_MESSAGE_NOT_FOUND, CAUSE_NOT_FOUND, null);
		}
	}

	/**
	 * Construye el mapa con los atributos de la incidencia para la respuesta a la petición de la operación genérica
	 *
	 * @param incidenceModel
	 * 		modelo de la incidencia a devolver
	 *
	 * @return el mapa con los datos de la incidencia
	 */
	private Map<String, FieldModel> buildResponseMap(IncidenceModel incidenceModel) {
		Map<String, FieldModel> map = new HashMap<>();

		map.put(FIELD_0, new FieldModel(incidenceModel.getIncidenceNumber()));
		map.put(FIELD_1, new FieldModel(incidenceModel.getGestionaRegistryNumber()));
		map.put(FIELD_2, new FieldModel(incidenceModel.getDni()));
		map.put(FIELD_3, new FieldModel(incidenceModel.getSubject()));

		return map;
	}

}
