package main.connectors.common.service;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Map;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.auth0.jwt.interfaces.DecodedJWT;

import main.connectors.common.config.ClientConfig;
import main.connectors.common.exception.SignatureErrorException;

@Service
public class SignatureService {

	private static final Logger log = LoggerFactory.getLogger(SignatureService.class);
	private final Map<String, String> clients;

	@Autowired
	private JwtService jwtService;

	@Autowired
	public SignatureService(ClientConfig clientConfig) {
		this.clients = clientConfig.getClientsMap();
	}

	/**
	 * Obtiene la id del cliente y el uuid del token y construye la firma
	 *
	 * @param token
	 * 		token JWT
	 *
	 * @return la firma a devolver en la respuesta
	 *
	 * @throws SignatureErrorException
	 * 		si no se ha podido calcular la firma
	 */
	public String obtainSignature(String token) throws SignatureErrorException {

		final DecodedJWT decodedToken = jwtService.decodeToken(token);
		final String clientId = jwtService.getClientToken(decodedToken);
		final String uuid = jwtService.getUuid(decodedToken);

		return buildSignature(clientId, uuid);
	}

	/**
	 * Construye la firma a partir del clientId y el uuid
	 *
	 * @param clientId
	 * 		id del cliente
	 * @param uuid
	 * 		uuid a firmar
	 *
	 * @return la firma a devolver en la respuesta
	 *
	 * @throws SignatureErrorException
	 * 		si no se ha podido calcular la firma
	 */
	private String buildSignature(String clientId, String uuid) throws SignatureErrorException {

		final String sharedKey = clients.get(clientId);

		try {
			final byte[] hash = mac256(uuid, sharedKey);
			final String hexaHash = toHexadecimal(hash);
			final String signature = encodeB64String(hexaHash);

			log.info("Firma del cliente con id {}: {}", clientId, signature);

			return signature;

		} catch (final Exception e) {
			throw new SignatureErrorException(String.format("No se ha podido calcular la firma: %s", e.getMessage()));
		}
	}

	private byte[] mac256(String stringToEncode, String sharedKey) throws Exception {
		// Se hace el MAC con la clave de la operación
		final Mac sha256HMAC = Mac.getInstance("HmacSHA256");
		final SecretKeySpec secretKey = new SecretKeySpec(sharedKey.getBytes(StandardCharsets.UTF_8), "HmacSHA256");
		sha256HMAC.init(secretKey);
		return sha256HMAC.doFinal(stringToEncode.getBytes(StandardCharsets.UTF_8));
	}

	private String toHexadecimal(byte[] data) throws IOException {
		final int numberOfBytes = data.length;
		final StringBuilder result = new StringBuilder();

		try (ByteArrayInputStream input = new ByteArrayInputStream(data, 0, numberOfBytes)) {
			int bytesRead = input.read();
			while (bytesRead != -1) {
				final String stringAux = Integer.toHexString(bytesRead);
				if (stringAux.length() < 2)// Hay que añadir un 0
					result.append("0");
				result.append(stringAux);
				bytesRead = input.read();
			}
			return result.toString();
		}
	}

	private String encodeB64String(String data) {
		return Base64.getEncoder().encodeToString(data.getBytes(StandardCharsets.UTF_8));
	}
}
