package main.connectors.common.service;

import static main.connectors.common.service.DataValidatorService.MSG_FIELD_UNEXPECTED_FORMAT;

import java.util.Map;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import jakarta.servlet.http.HttpServletRequest;
import main.connectors.common.exception.SignatureErrorException;
import main.connectors.common.model.ErrorModel;
import main.connectors.common.model.FieldModel;
import main.connectors.common.response.BookmarkResponse;
import main.connectors.common.response.VersionResponse;

@Service
public class ConnectorUtilsService {

	public static final String CONTENT_TYPE = "Content-Type";
	public static final String SIGNATURE_HEADER = "Signature";

	public static final String CONNECTOR_VERSION = "1.0";

	public static final String FIELD_0 = "FIELD_0";
	public static final String FIELD_1 = "FIELD_1";
	public static final String FIELD_2 = "FIELD_2";
	public static final String FIELD_3 = "FIELD_3";
	public static final String FIELD_4 = "FIELD_4";
	public static final String FIELD_5 = "FIELD_5";
	public static final String FIELD_6 = "FIELD_6";
	public static final String FIELD_7 = "FIELD_7";
	public static final String FIELD_8 = "FIELD_8";
	public static final String FIELD_9 = "FIELD_9";

	public static final String CAUSE_FIELD_ERRORS = "FIELD_ERRORS";
	public static final String MESSAGE_FIELD_ERRORS = "There are errors in fields";

	public static final String CAUSE_NOT_FOUND = "ELEMENT_NOT_EXISTS";
	public static final String INCIDENCE_MESSAGE_NOT_FOUND = "Incidence not found";

	public static final String MEDIA_TYPE_ERROR = "application/vnd.generic-operation.error+json";
	public static final String MEDIA_TYPE_GENERIC_OP = "application/vnd.generic-operation-response+json";
	public static final String MEDIA_TYPE_BOOKMARK = "application/vnd.bookmark-list+json";
	public static final String MEDIA_TYPE_VERSION = "application/vnd.version+json";
	private static final String SERVER_URL_FORMAT = "%s://%s:%d";

	@Autowired
	private SignatureService signatureService;

	@Autowired
	private DataValidatorService dataValidatorService;

	/**
	 * Crea las cabeceras para la respuesta a la petición
	 *
	 * @param contentType
	 * 		el tipo del contenido a devolver
	 * @param token
	 * 		el token recibido en la petición
	 *
	 * @return las cabeceras
	 *
	 * @throws SignatureErrorException
	 * 		si no se ha podido calcular la firma
	 */
	public HttpHeaders createHeaders(String contentType, String token) throws SignatureErrorException {
		final HttpHeaders headers = new HttpHeaders();
		headers.add(CONTENT_TYPE, contentType);

		if (token != null) {
			headers.add(SIGNATURE_HEADER, signatureService.obtainSignature(token));
		}

		return headers;
	}

	/**
	 * Obtiene la url al servidor donde se ha levantado la aplicación
	 *
	 * @param request
	 * 		la petición enviada
	 *
	 * @return la url del servidor
	 */
	public String obtainServerUrl(HttpServletRequest request) {
		return String.format(SERVER_URL_FORMAT, request.getScheme(), request.getServerName(), request.getServerPort());
	}

	/**
	 * Genera la respuesta al recurso de versión
	 *
	 * @param token
	 * 		el token para generar la firma
	 *
	 * @return la respuesta a la petición
	 */
	public ResponseEntity<VersionResponse> generateVersionResponse(String token) {
		final HttpHeaders headers = createHeaders(MEDIA_TYPE_VERSION, token);
		final VersionResponse versionResponse = new VersionResponse();

		return new ResponseEntity<>(versionResponse, headers, HttpStatus.OK);
	}

	/**
	 * Genera la respuesta al recurso bookmark
	 *
	 * @param bookmark
	 * 		la url al recurso de operación genérica
	 * @param token
	 * 		el token para generar la firma
	 *
	 * @return la respuesta a la petición
	 */
	public ResponseEntity<BookmarkResponse> generateBookmarkResponse(String bookmark, String token) {

		final HttpHeaders headers = createHeaders(MEDIA_TYPE_BOOKMARK, token);
		final BookmarkResponse bookmarkResponse = new BookmarkResponse(bookmark);

		return new ResponseEntity<>(bookmarkResponse, headers, HttpStatus.OK);
	}

	/**
	 * Comprueba si existen errores en los datos recibidos en la petición y en el caso de que así sea devuelve la respuesta correspondiente a tal situación
	 *
	 * @param data
	 * 		datos recibidos en la petición
	 * @param expectedFields
	 * 		campos esperados en la petición
	 *
	 * @return la respuesta a la petición en el caso de error, o null en el caso contrario
	 */
	public ResponseEntity<ErrorModel> checkErrors(Map<String, FieldModel> data, Set<String> expectedFields) {
		return checkErrors(data, expectedFields, false);
	}

	/**
	 * Comprueba si existen errores en los datos recibidos en la petición y en el caso de que así sea devuelve la respuesta correspondiente a tal situación.
	 * Además, si se indica el `checkPriority`, valida el formato del campo prioridad
	 *
	 * @param data
	 * 		datos recibidos en la petición
	 * @param expectedFields
	 * 		campos esperados en la petición
	 * @param checkPriority
	 * 		si es verdadero, comprueba el formato del campo prioridad, en caso contrario, realiza la comprobación normal
	 *
	 * @return la respuesta a la petición en el caso de error, o null en el caso contrario
	 */
	public ResponseEntity<ErrorModel> checkErrors(Map<String, FieldModel> data, Set<String> expectedFields, boolean checkPriority) {
		// Sacamos el mapa de errores
		final Map<String, String> errors = dataValidatorService.checkDataParameters(expectedFields, data);

		// En el caso de que haya que comprobar el valor de la prioridad
		if (checkPriority) {
			checkPriorityValue(data, errors);
		}

		// En el caso de que haya errores, generamos y devolvemos el error
		if (!errors.isEmpty()) {
			return buildErrorResponse(MESSAGE_FIELD_ERRORS, CAUSE_FIELD_ERRORS, errors);
		}

		return null;
	}

	/**
	 * Construye un error de respuesta genérica
	 *
	 * @param message
	 * 		mensaje del error
	 * @param cause
	 * 		causa del error
	 * @param data
	 * 		datos adicionales del error (en caso de que lo haya)
	 *
	 * @return ResponseEntity con el error de respuesta genérica
	 */
	public ResponseEntity<ErrorModel> buildErrorResponse(String message, String cause, Map<String, String> data) {

		final ErrorModel errorModel = new ErrorModel(message, cause, data);
		final HttpHeaders headers = createHeaders(MEDIA_TYPE_ERROR, null);

		return new ResponseEntity<>(errorModel, headers, HttpStatus.PRECONDITION_FAILED);
	}

	/**
	 * Comprueba si el campo prioridad recibido es válido, es decir, si es un entero y se encuentra entre 1 y 10, ambos incluidos
	 *
	 * @param data
	 * 		datos recibidos de la petición
	 * @param errors
	 * 		errores encontrados antes de comprobar el campo prioridad
	 */
	private void checkPriorityValue(Map<String, FieldModel> data, Map<String, String> errors) {

		// Si el campo 1 ya se encuentra en los errores no es necesaria la comprobación
		if (errors.containsKey(FIELD_1)) {
			return;
		}

		try {
			final int priorityField = Integer.parseInt(data.get(FIELD_1).getValue());
			if (priorityField < 1 || priorityField > 10) {
				errors.put(FIELD_1, MSG_FIELD_UNEXPECTED_FORMAT);
			}
		} catch (NumberFormatException e) {
			errors.put(FIELD_1, MSG_FIELD_UNEXPECTED_FORMAT);
		}
	}
}
