package main.connectors.common.config;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.core.env.Environment;
import org.springframework.validation.annotation.Validated;

import jakarta.annotation.PostConstruct;
import jakarta.validation.constraints.NotEmpty;

@Validated
@ConfigurationProperties(prefix = "mockup.connectors.clients")
public class ClientConfig {

	private static final Logger log = LoggerFactory.getLogger(ClientConfig.class);

	private static final String CLIENT_ID_FORMAT = "mockup.connectors.clients.%s.id";
	private static final String CLIENT_PUBLIC_KEY_FORMAT = "mockup.connectors.clients.%s.public-key";

	@NotEmpty(message = "The list of clients must not be empty")
	private List<String> list;

	private Map<String, String> clientsMap;

	@Autowired
	private Environment env;

	public List<String> getList() {
		return list;
	}

	public void setList(List<String> list) {
		this.list = list;
	}

	@PostConstruct
	public void init() {
		clientsMap = new HashMap<>();

		for (String clientName : list) {

			final String id = env.getProperty(String.format(CLIENT_ID_FORMAT, clientName));
			final String publicKey = env.getProperty(String.format(CLIENT_PUBLIC_KEY_FORMAT, clientName));

			if (id == null || publicKey == null) {
				throw new IllegalArgumentException(String.format("Missing properties for client '%s': id or public key is not defined.", clientName));
			}

			log.info("Loading client '{}' with ID: {}, Public Key: {}", clientName, id, publicKey);
			clientsMap.put(id, publicKey);
		}
	}

	public Map<String, String> getClientsMap() {
		return clientsMap;
	}
}
