using mockup_gestiona_for_developers_net.Exception;
using mockup_gestiona_for_developers_net.Models;
using mockup_gestiona_for_developers_net.Repository;

namespace mockup_gestiona_for_developers_net.Service;

using Configuration;
using Connectors.ThirdData.Services;
using Gestiona.ApiRest;

public class IncidenceService(
    IIncidenceRepository incidenceRepository,
    GestionaApiService gestionaApiService,
    MockupSettings mockupSettings,
    GoogleSheetsService googleSheetsService)
{

    private readonly string _procedureName = mockupSettings.GestionaApiSettings.ProcedureSettings.ProcedureName;
    private readonly string _externalProcedureName = mockupSettings.GestionaApiSettings.ProcedureSettings.ExternalProcedureName;

    /// <summary>
    /// Busca una incidencia en base de datos a través de su número de expediente
    /// </summary>
    /// <param name="numFile">Número de expediente</param>
    /// <returns>El modelo de la incidencia en el caso de que la haya encontrado</returns>
    /// <exception cref="IncidenceNotFoundException">En el caso de que no haya encontrado la incidencia en base de datos</exception>
    public async Task<IncidenceModel> SearchIncidence(string numFile)
    {
        var incidence = await incidenceRepository.GetIncidenceByNumFile(numFile);

        if (incidence == null)
        {
            throw new IncidenceNotFoundException($"No se ha encontrado la incidencia con número de expediente {numFile}");
        }

        return incidence;
    }

    /// <summary>
    /// Obtiene la incidencia de base de datos a partir de su número de incidencia
    /// </summary>
    /// <param name="numIncidence">Número de incidencia</param>
    /// <returns>El modelo de la incidencia en el caso de que la haya encontrado</returns>
    /// <exception cref="IncidenceNotFoundException">En el caso de que no haya encontrado la incidencia en base de datos</exception>
    public async Task<IncidenceModel> GetIncidenceByNumber(string numIncidence)
    {
        var incidence = await incidenceRepository.GetIncidenceByNumber(numIncidence);

        if (incidence == null)
        {
            throw new IncidenceNotFoundException($"No se ha encontrado la incidencia con el número {numIncidence}");
        }

        return incidence;
    }

    /// <summary>
    /// Obtiene los datos de los terceros para poder sacarlos por la GUI
    /// </summary>
    /// <returns>La matriz con todos los datos de los terceros</returns>
    public async Task<IList<IList<object>>?> GetThirds()
    {
        return await googleSheetsService.GetAllThirds();
    }

    /// <summary>
    /// Registra una nueva incidencia en Gestiona y finalmente almacena los datos de esta en la base de datos
    /// </summary>
    /// <param name="nif">DNI del tercero a asignar</param>
    /// <param name="subject">Asunto de la incidencia</param>
    public async Task RegisterNewIncidence(string nif, string subject)
    {
        await gestionaApiService.CheckIfThirdExists(nif);

        var registry = await gestionaApiService.CreateInputRegistry(subject);

        var thirdParty = await gestionaApiService.GetThirdPartyByNif(nif);
        await gestionaApiService.AddThirdPartyToInputRegistry(registry, thirdParty);

        await gestionaApiService.FinalizeRegistryAnnotation(registry);

        var refreshedRegistry = await gestionaApiService.ObtainRegistryAnnotation(registry);

        var file = await gestionaApiService.CreateFileWithProcedure(this._procedureName, this._externalProcedureName);

        await gestionaApiService.AddRegistryToFile(file, refreshedRegistry);

        var incidence = new IncidenceModel(nif, subject, registry.Code, file.Code, IncidenceModel.IncidenceStatus.UNASSIGNED);

        await incidenceRepository.AddIncidenceAsync(incidence);
    }

    /// <summary>
    /// Marca la incidencia como ASSIGNED
    /// </summary>
    /// <param name="incidenceModel">Modelo de la incidencia a modificar</param>
    /// <param name="priority">Prioridad de la incidencia a establecer</param>
    public async Task SetAssignedIncidence(IncidenceModel incidenceModel, int priority)
    {
        incidenceModel.Status = IncidenceModel.IncidenceStatus.ASSIGNED;
        incidenceModel.Priority = priority;
        await incidenceRepository.SaveAsync(incidenceModel);
    }

    /// <summary>
    /// Marca la incidencia como RESOLVED
    /// </summary>
    /// <param name="incidenceModel">Modelo de la incidencia a modificar</param>
    /// <param name="setToResolved">Flag indicando si hay que marcar la incidencia a RESOLVED</param>
    public async Task SetResolvedIncidence(IncidenceModel incidenceModel, bool setToResolved)
    {
        if (!setToResolved) return;

        incidenceModel.Status = IncidenceModel.IncidenceStatus.RESOLVED;
        await incidenceRepository.SaveAsync(incidenceModel);
    }
}