namespace mockup_gestiona_for_developers_net.Service;

using Configuration;
using Connectors.ThirdData.Services;
using Gestiona.ApiRest;
using Gestiona.ApiRest.Exceptions;
using Gestiona.ApiRest.Models;
using Models;
using Repository;
using Exception = System.Exception;

public class DocumentaryFlowService(
    IDocumentaryFlowRepository documentaryFlowRepository,
    GestionaApiService gestionaApiService,
    GoogleSheetsService googleSheetsService,
    ILogger<DocumentaryFlowService> logger,
    MockupSettings mockupSettings)
{

    private static readonly Action<ILogger, string, Exception?> LogErrorFetchingDocumentFlowStateAction =
        LoggerMessage.Define<string>(
            LogLevel.Error,
            new EventId(1004, nameof(LogErrorFetchingDocumentFlowState)),
            "{Message}");

    private readonly string _fileUploadPath = mockupSettings.UploadsSettings.FileUploadPath;

    /// <summary>
    /// Sincroniza los flujos documentales en la base de datos con los expedientes obtenidos de Gestiona.
    /// En el caso de que un expediente no tenga asociado un flujo documental, se va a crear uno nuevo.
    /// </summary>
    public async Task DocumentaryFlowDatabaseSync()
    {
        var fileModelList = await gestionaApiService.GetFileList();

        foreach (var fileModel in fileModelList)
        {
            var hasDocumentaryFlow = await HasDocumentaryFlow(fileModel);
            if (hasDocumentaryFlow) continue;

            var newModel = new DocumentaryFlowModel(fileModel.Code, fileModel.Id);
            await documentaryFlowRepository.AddDocumentaryFlowAsync(newModel);
        }
    }

    /// <summary>
    /// Actualiza aquellos flujos documentales cuyos documentos ya hayan sido tramitados
    /// </summary>
    public async Task UpdateProcessStatus()
    {
        var documentaryFlowList = await documentaryFlowRepository.GetAllDocumentaryFlowsAsync();

        foreach (var documentaryFlow in documentaryFlowList)
        {
            try
            {
                if (!DocumentaryFlowModel.Status.IN_PROCESS.Equals(documentaryFlow.State)) continue;
                var documentProcessStateModel = await gestionaApiService.CheckDocumentProcessStatus(documentaryFlow.DocumentReference!);

                var state = documentProcessStateModel.State;
                if (!DocumentProcessStateModel.DocumentProcessState.IN_PROGRESS.Equals(state))
                {
                    await SetDocumentFlowToFinalState(documentaryFlow, state);
                }
            }
            catch (Exception e)
            {
                LogErrorFetchingDocumentFlowState($"Error al obtener el estado del flujo documental: {e}", e);
            }
        }
    }

    /// <summary>
    /// Gestiona la lógica para la subida del documento a Gestiona
    /// </summary>
    /// <param name="documentaryFlowId">Id del flujo documental</param>
    /// <param name="content">Contenido del documento a subir</param>
    public async Task HandleDocumentUpload(string documentaryFlowId, MultipartFormDataContent content)
    {
        try
        {
            var documentaryFlow = await documentaryFlowRepository.FindById(documentaryFlowId);
            var fileModel = await FindFileModel(documentaryFlow.FileIdGestiona!);

            var path = await SaveUploadedFileAsync(content);

            var documentReference = await gestionaApiService.AddDocumentToFile(fileModel, path);

            await SetDocumentFlowToUploaded(documentaryFlow, documentReference);
        }
        catch (Exception e)
        {
            LogErrorFetchingDocumentFlowState($"Error al tratar de subir el documento: {e}", e);
        }
    }

    /// <summary>
    /// Obtiene los circuitos de tramitación disponibles para un documento
    /// </summary>
    /// <param name="documentaryFlowId">Id del flujo documental que tiene el documento asociado</param>
    /// <returns>Listado de datos de circuitos de tramitación disponibles para el documento</returns>
    public async Task<List<CircuitTemplateModel>> GetCircuitTemplatesForDocument(string documentaryFlowId)
    {
        var documentaryFlow = await documentaryFlowRepository.FindById(documentaryFlowId);

        return await gestionaApiService.ObtainCircuitTemplates(documentaryFlow.DocumentReference!);
    }

    /// <summary>
    /// Obtiene los datos de los terceros para poder sacarlos por la GUI
    /// </summary>
    /// <returns>La matriz con todos los datos de los terceros</returns>
    public async Task<IList<IList<object>>?> GetThirds()
    {
        return await googleSheetsService.GetAllThirds();
    }

    /// <summary>
    /// Maneja la la tramitación de un documento con un circuito de tramitación y un tercero al que notificar
    /// </summary>
    /// <param name="documentaryFlowId">Id del flujo documental</param>
    /// <param name="circuitTemplateModel">Datos del circuito de tramitación a aplicar</param>
    /// <param name="nif">NIF del tercero a notificar</param>
    public async Task ProcessDocument(string documentaryFlowId, CircuitTemplateModel circuitTemplateModel, string nif)
    {
        try
        {
            var documentaryFlow = await documentaryFlowRepository.FindById(documentaryFlowId);

            if (circuitTemplateModel.SendRegistryOutput)
            {
                await gestionaApiService.CheckIfThirdExists(nif);
                var third = await gestionaApiService.GetThirdOutputByNif(nif);

                var thirdOutputModelList = circuitTemplateModel.ThirdsOutput ?? [];

                thirdOutputModelList.Add(third);
                circuitTemplateModel.ThirdsOutput = thirdOutputModelList;
            }

            await gestionaApiService.ProcessDocument(documentaryFlow.DocumentReference!, circuitTemplateModel);
            await SetDocumentFlowToInProcess(documentaryFlow);
        }
        catch (Exception e)
        {
            LogErrorFetchingDocumentFlowState($"Error al tratar de procesar el documento: {e}", e);
        }
    }

    /// <summary>
    /// Maneja la lógica necesaria para obtener el contenido del documento tramitado
    /// </summary>
    /// <param name="documentaryFlowId">Id del flujo documental cuyo documento tramitado se quiere descargar</param>
    /// <returns>El contenido del documento y el nombre completo con su extensión</returns>
    public async Task<(byte[], string fileName)> DownloadDocument(string documentaryFlowId)
    {
        var documentaryFlow = await documentaryFlowRepository.FindById(documentaryFlowId);

        var documentModel = await gestionaApiService.GetDocument(documentaryFlow.DocumentReference!);

        var documentContent = await gestionaApiService.GetDocumentContent(documentModel);

        return (documentContent, documentModel.FileNameWithExtension!);
    }

    /// <summary>
    /// Comprueba si un expediente tiene asociado algún flujo documental
    /// </summary>
    /// <param name="fileModel">Datos del expediente a buscar</param>
    /// <returns>Devuelve true en el caso de que sí que tenga un flujo documental asociado, false en caso contrario</returns>
    private async Task<bool> HasDocumentaryFlow(FileModel fileModel)
    {
        return await documentaryFlowRepository.FindByFileIdGestiona(fileModel.Id!) != null;
    }

    /// <summary>
    /// Obtiene los datos de un expediente a partir de su id
    /// </summary>
    /// <param name="fileId">Id del expediente a buscar</param>
    /// <returns>Los datos del expediente si los encuentra, null en caso contrario</returns>
    private async Task<FileModel> FindFileModel(string fileId)
    {
        var fileModelList = await gestionaApiService.GetFileList();

        return fileModelList.FirstOrDefault(fm => fm.Id!.Equals(fileId, StringComparison.Ordinal)) ?? throw new InvalidOperationException();
    }

    /// <summary>
    /// Almacena el fichero en la ruta indicada en el servidor
    /// </summary>
    /// <param name="content">Contenido subido por el usuario a través de la GUI</param>
    /// <returns>La ruta absoluta en la que se almacena el fichero</returns>
    /// <exception cref="InvalidOperationException"></exception>
    /// <exception cref="Exception"></exception>
    private async Task<string> SaveUploadedFileAsync(MultipartFormDataContent content)
    {
        if (!Directory.Exists(this._fileUploadPath))
        {
            Directory.CreateDirectory(this._fileUploadPath);
        }

        foreach (var httpContent in content)
        {
            if (httpContent is not StreamContent streamContent) continue;

            var fileName = streamContent.Headers.ContentDisposition?.FileName?.Trim('"');

            if (string.IsNullOrEmpty(fileName))
            {
                throw new InvalidOperationException("El archivo no tiene un nombre válido");
            }

            var filePath = Path.Combine(this._fileUploadPath, fileName);

            await using var fileStream = new FileStream(filePath, FileMode.Create, FileAccess.Write, FileShare.None);
            await streamContent.CopyToAsync(fileStream);

            return filePath;
        }

        throw new InvalidOperationException("No se ha podido almacenar el archivo en el servidor");
    }

    /// <summary>
    /// Modifica el estado de un flujo documental a UPLOADED y almacena la referencia al documento asociado en Gestiona
    /// </summary>
    /// <param name="documentaryFlow">Datos del flujo documental</param>
    /// <param name="documentReference">Referencia al documento asociado en Gestiona</param>
    private async Task SetDocumentFlowToUploaded(DocumentaryFlowModel documentaryFlow, string documentReference)
    {
        documentaryFlow.DocumentReference = documentReference;
        documentaryFlow.State = DocumentaryFlowModel.Status.UPLOADED;
        await documentaryFlowRepository.UpdateDocumentaryFlowAsync(documentaryFlow);
    }

    /// <summary>
    /// Modifica el estado del flujo documental a IN_PROCESS
    /// </summary>
    /// <param name="documentaryFlow">Datos del flujo documental</param>
    private async Task SetDocumentFlowToInProcess(DocumentaryFlowModel documentaryFlow)
    {
        documentaryFlow.State = DocumentaryFlowModel.Status.IN_PROCESS;
        await documentaryFlowRepository.UpdateDocumentaryFlowAsync(documentaryFlow);
    }

    /// <summary>
    /// Modifica el estado del flujo documental a PROCESSED y rellena los datos del registro al que se ha asociado
    /// </summary>
    /// <param name="documentaryFlow">Datos del flujo documental</param>
    /// <param name="state">Estado de la tramitación</param>
    private async Task SetDocumentFlowToFinalState(DocumentaryFlowModel documentaryFlow, DocumentProcessStateModel.DocumentProcessState state)
    {
        var fileDocumentAnnotation = await gestionaApiService.GetFileDocumentAnnotationModel(documentaryFlow.DocumentReference!);


        // En el caso de que no tenga una anotación asociada se informará su estado como `UNKNOWN_ERROR`
        var documentaryFlowNewState = DocumentaryFlowModel.Status.UNKNOWN_ERROR;

        if (fileDocumentAnnotation != null)
        {
            documentaryFlow.RegistryCodeGestiona = fileDocumentAnnotation.Code;
            documentaryFlow.RegistryIdGestiona = fileDocumentAnnotation.Id;
            documentaryFlowNewState = Enum.Parse<DocumentaryFlowModel.Status>(state.ToString());
        }

        documentaryFlow.State = documentaryFlowNewState;

        await documentaryFlowRepository.UpdateDocumentaryFlowAsync(documentaryFlow);
    }

    private void LogErrorFetchingDocumentFlowState(string message, Exception exception)
    {
        LogErrorFetchingDocumentFlowStateAction(logger, message, exception);
    }

}