namespace mockup_gestiona_for_developers_net.Gestiona.ApiRest.Services;

using System.Net;
using Exceptions;
using Exception = System.Exception;

public class UploadService(GestionaApiUtilsService gestionaApiUtilsService)
{

    /// <summary>
    /// Crea una nueva entrada en el recurso uploads de Gestiona
    /// </summary>
    /// <param name="uploadReference">Url sobre la cual realizar la petición</param>
    /// <returns>La referencia al recurso generado</returns>
    /// <exception cref="GestionaApiException">En caso de no poder crear la entrada en el recurso uploads</exception>
    public async Task<string> CreateEmptyUpload(string uploadReference)
    {
        try
        {
            var response = await gestionaApiUtilsService.CreatePetitionAsync<string>(uploadReference, HttpMethod.Post, null, null);

            return response.Headers.Location!.ToString();
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"No se ha podido crear el upload vacío {e}");
        }
    }

    /// <summary>
    /// Sube el contenido de un documento al recurso uploads definido por parámetro
    /// </summary>
    /// <param name="uploadReference">Url al recurso uploads</param>
    /// <param name="documentPath">Ruta local del documento a subir</param>
    /// <returns>El nombre del documento subido</returns>
    /// <exception cref="GestionaApiException">En caso de no poder subir el documento</exception>
    public async Task<string> UploadDocument(string uploadReference, string documentPath)
    {
        try
        {
            if (!File.Exists(documentPath))
                throw new GestionaApiException($"El archivo no existe: {documentPath}");

            var response = await gestionaApiUtilsService.UploadDocumentPetitionAsync(uploadReference,
                new ByteArrayContent(await File.ReadAllBytesAsync(documentPath)));

            if (response.StatusCode != HttpStatusCode.OK)
            {
                throw new GestionaApiException($"No se ha podido subir el documento {documentPath} al upload {uploadReference}");
            }

            return GetFileNameWithoutExtension(Path.GetFileName(documentPath));
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"No se ha podido subir el documento {documentPath} al upload {uploadReference}: {e}");
        }
    }

    /// <summary>
    /// Saca el nombre del fichero sin su extensión
    /// </summary>
    /// <param name="fileName">Nombre completo del fichero</param>
    /// <returns>La cadena resultante de eliminarle la extensión</returns>
    private static string GetFileNameWithoutExtension(string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
        {
            return string.Empty;
        }

        var lastDotIndex = fileName.LastIndexOf('.');

        return lastDotIndex == -1 ? fileName : fileName[..lastDotIndex];
    }
}