namespace mockup_gestiona_for_developers_net.Gestiona.ApiRest.Services;

using System.Diagnostics;
using System.Net;
using System.Text;
using Connectors.ThirdData.Services;
using Exceptions;
using Models;
using Exception = System.Exception;

public class ThirdService(GestionaApiUtilsService gestionaApiUtilsService, GoogleSheetsService googleSheetsService)
{

    private static readonly CompositeFormat FilterViewFormat = CompositeFormat.Parse("{0}?filter-view={1}");
    private const string THIRDS_BOOKMARK = "vnd.gestiona.thirds";
    private const string THIRD_REL = "third";
    private const string MEDIA_TYPE_THIRD_PARTY = "application/vnd.gestiona.thirdparty+json;version=3";
    private const string MEDIA_TYPE_THIRD = "application/vnd.gestiona.third+json; version=3";
    private const int SHEETS_NAME_COLUMN = 2;
    private const string NIF_TYPE = "NIF";
    private const string NIF_COUNTRY = "ESP";
    private const string PHISIC_TYPE = "PHISIC";

    /// <summary>
    /// Obtiene los datos de un tercero a través de su NIF
    /// </summary>
    /// <param name="nif">Nif del tercero a buscar</param>
    /// <returns>Los datos del tercero si lo encuentra</returns>
    /// <exception cref="GestionaApiException">Si no es capaz de encontrar al tercero en Gestiona</exception>
    public async Task<ThirdModel> GetThirdByNif(string nif)
    {
        try
        {
            var url = gestionaApiUtilsService.GetBookmarkUrl(THIRDS_BOOKMARK);
            var filterUrl = CreateFilterUrl(url, new ThirdFilterModel(nif));

            var response = await gestionaApiUtilsService.CreatePetitionAsync<ThirdPageModel>(filterUrl, HttpMethod.Get, null, null);

            if (response.StatusCode.Equals(HttpStatusCode.NoContent))
            {
                throw new GestionaApiException($"No se ha encontrado ningún tercero con NIF {nif}");
            }

            return response.Body!.Content!.First();
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"Error obteniendo los datos del tercero: {e}");
        }
    }

    /// <summary>
    /// Crea los datos de salida de un tercero a partir de sus datos
    /// </summary>
    /// <param name="thirdModel">Datos del tercero</param>
    /// <returns>Los datos de salida del tercero</returns>
    public static ThirdOutputModel CreateThirdOutputModel(ThirdModel thirdModel)
    {
        var thirdOutputModel = new ThirdOutputModel();

        var notificationChannel = thirdModel.NotificationChannel;
        thirdOutputModel.NotificationChannel = notificationChannel;
        thirdOutputModel.Relation = ThirdOutputModel.RelationType.INVOLVED;

        var thirdLink = new LinkModel(THIRD_REL, GestionaApiUtilsService.ObtainLinkByRel(thirdModel.Links!, GestionaApiUtilsService.SELF_REL));
        thirdOutputModel.AddLink(thirdLink);

        return thirdOutputModel;
    }

    /// <summary>
    /// Obtiene la dirección por defecto de un tercero 
    /// </summary>
    /// <param name="url">Url al recurso de la dirección por defecto de un tercero</param>
    /// <returns>Los datos de la dirección del tercero</returns>
    /// <exception cref="GestionaApiException">En caso de que no haya podido obtener la dirección del tercero</exception>
    public async Task<ThirdAddressModel> GetThirdAddressModel(string url)
    {
        try
        {
            var response = await gestionaApiUtilsService.CreatePetitionAsync<ThirdAddressModel>(url, HttpMethod.Get, null, null);
            return response.Body ?? throw new GestionaApiException("Error obteniendo la dirección del tercero.");
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"Error obteniendo la dirección del tercero: {e}");
        }
    }

    /// <summary>
    /// Añade a los datos de salida del tercero los datos necesarios en el caso de que su canal de notificación sea PAPER
    /// </summary>
    /// <param name="thirdOutputModel">Datos de salida del tercero</param>
    /// <param name="thirdAddressModel">Datos de la dirección del tercero</param>
    public static void AddPaperNecessaryFields(ThirdOutputModel thirdOutputModel, ThirdAddressModel thirdAddressModel)
    {
        thirdOutputModel.Address = thirdAddressModel.Address;
        thirdOutputModel.Country = thirdAddressModel.Country;
        thirdOutputModel.Zip = thirdAddressModel.ZipCode;
        thirdOutputModel.Zone = thirdAddressModel.Zone;
    }

    /// <summary>
    /// Añade un tercero a un registro de entrada
    /// </summary>
    /// <param name="url">Url sobre la que hacer la petición para añadirlo</param>
    /// <param name="thirdPartyModel">Datos del tercero asociado al registro</param>
    /// <exception cref="GestionaApiException">En caso de que no pueda añadir el tercero al registro</exception>
    public async Task AddThirdPartyToInputRegistry(string url, ThirdPartyModel thirdPartyModel)
    {
        try
        {
            var extraHeaders = new Dictionary<string, string> { { GestionaApiUtilsService.CONTENT_TYPE, MEDIA_TYPE_THIRD_PARTY } };

            var response = await gestionaApiUtilsService.CreatePetitionAsync<object>(url, HttpMethod.Post, extraHeaders, thirdPartyModel);

            if (!response.StatusCode.Equals(HttpStatusCode.Created))
            {
                throw new GestionaApiException("No se ha podido añadir el tercero al registro");
            }
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"Error añadiendo el tercero al registro: {e}");
        }
    }

    /// <summary>
    /// Crea los datos de un tercero asociado a partir de los datos del tercero
    /// </summary>
    /// <param name="thirdModel">Datos del tercero</param>
    /// <returns>Los datos del tercero asociado</returns>
    public static ThirdPartyModel CreateThirdPartyModel(ThirdModel thirdModel)
    {
        return new ThirdPartyModel(thirdModel);
    }

    /// <summary>
    /// Crea un nuevo tercero en Gestiona a partir de su NIF y de los datos de la hoja de Google Sheets
    /// </summary>
    /// <param name="nif"></param>
    /// <exception cref="GestionaApiException"></exception>
    public async Task CreateThird(string nif)
    {
        try
        {
            var thirdModel = await CreateThirdModel(nif);

            var url = gestionaApiUtilsService.GetBookmarkUrl(THIRDS_BOOKMARK);

            var extraHeaders = new Dictionary<string, string> { { GestionaApiUtilsService.CONTENT_TYPE, MEDIA_TYPE_THIRD } };

            var response = await gestionaApiUtilsService.CreatePetitionAsync<ThirdModel>(url, HttpMethod.Post, extraHeaders, thirdModel);

            if (!response.StatusCode.Equals(HttpStatusCode.Created))
            {
                throw new GestionaApiException($"No se ha podido crear el tercero con NIF: {nif}");
            }
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"Error creando el tercero con NIF {nif}: {e}");
        }
    }

    /// <summary>
    /// Construye los datos de un tercero a partir de su nif y los datos de la hoja de Google Sheets
    /// </summary>
    /// <param name="nif">Nif del tercero de Google Sheets a buscar en la hoja</param>
    /// <returns>Los datos del tercero creados</returns>
    private async Task<ThirdModel> CreateThirdModel(string nif)
    {
        var thirdData = await googleSheetsService.GetThirdByNifAsync(nif);

        Debug.Assert(thirdData != null, nameof(thirdData) + " != null");

        var thirdModel = new ThirdModel
        {
            Nif = nif,
            FullName = thirdData[SHEETS_NAME_COLUMN].ToString(),
            NifType = NIF_TYPE,
            NifCountry = NIF_COUNTRY,
            Type = PHISIC_TYPE,
            NotificationChannel = GestionaApiUtilsService.NotificationChannelType.TELEMATIC
        };

        var splitName = thirdData[SHEETS_NAME_COLUMN].ToString()!.Split(' ');
        thirdModel.FirstName = splitName[0];
        thirdModel.FirstSurname = splitName[1];

        if (splitName.Length > 2)
        {
            thirdModel.SecondSurname = splitName[2];
        }

        return thirdModel;
    }

    /// <summary>
    /// Añade el filtro `filter-view` a la url con el nif codificado
    /// </summary>
    /// <param name="url">Url a la que añadirle la query</param>
    /// <param name="thirdFilterModel">Datos del filtro</param>
    /// <returns>La url con la query concatenada</returns>
    private static string CreateFilterUrl(string url, ThirdFilterModel thirdFilterModel)
    {
        var bytes = Encoding.UTF8.GetBytes(thirdFilterModel.ToString());

        var encodedUrlSafe = Convert.ToBase64String(bytes)
            .Replace("+", "-")
            .Replace("/", "_")
            .TrimEnd('=');

        return string.Format(null, FilterViewFormat, url, encodedUrlSafe);
    }
}