namespace mockup_gestiona_for_developers_net.Gestiona.ApiRest.Services;

using System.Net;
using Configuration;
using Exceptions;
using Models;
using Exception = System.Exception;

public class RegistryService(GestionaApiUtilsService gestionaApiUtilsService, MockupSettings mockupSettings)
{
    private const string MEDIA_TYPE_REGISTRY_ANNOTATION = "application/vnd.gestiona.registry-annotation+json;version=2";
    private const string REGISTRY_OFFICES_REL = "vnd.gestiona.registry.offices";
    private const string MEDIA_TYPE_LINKS = "application/vnd.gestiona.links+json";
    private const string SEND_TO_FILE_REL = "send-to-file";
    private const string FINALIZE_REL = "finalize";
    private const string FILE_REL = "file";
    private const string DOCUMENTS_REL = "documents";
    private const string INPUT_LINK = "input";

    private readonly string _registryOfficeCode = mockupSettings.GestionaApiSettings.RegistriesSettings.RegistryOfficeCode;

    private readonly string _registryCategory = mockupSettings.GestionaApiSettings.RegistriesSettings.CategoryName;

    private readonly string _registrySubcategory = mockupSettings.GestionaApiSettings.RegistriesSettings.SubcategoryName;

    /// <summary>
    /// Crea un registro de entrada en Gestiona con el resumen indicado
    /// </summary>
    /// <param name="summary">Resumen del registro que se va a crear</param>
    /// <returns>Los datos de la anotación creada en Gestiona</returns>
    /// <exception cref="GestionaApiException">En el caso de que no sea capaz de crear el registro</exception>
    public async Task<RegistryAnnotationModel> CreateInputRegistry(string summary)
    {
        try
        {
            var url = ObtainInputRegistryOfficeLink(await GetRegistryOfficeModels());

            var inputRegistryModel = new RegistryAnnotationModel(summary, this._registryCategory, this._registrySubcategory);

            var extraHeaders = new Dictionary<string, string> { { GestionaApiUtilsService.CONTENT_TYPE, MEDIA_TYPE_REGISTRY_ANNOTATION } };

            var response = await gestionaApiUtilsService.CreatePetitionAsync<RegistryAnnotationModel>(url, HttpMethod.Post, extraHeaders, inputRegistryModel);

            if (response.StatusCode != HttpStatusCode.Created) throw new GestionaApiException("Error al crear el registro de entrada");

            var locationUrl = response.Headers.Location;
            return await ObtainRegistryAnnotationModel(locationUrl!.ToString());
        }
        catch (Exception ex)
        {
            throw new GestionaApiException($"Error al crear el registro de entrada: {ex}");
        }
    }

    /// <summary>
    /// Añade un registro de entrada a un expediente
    /// </summary>
    /// <param name="fileModel">Datos del expediente</param>
    /// <param name="registryAnnotationModel">Datos del registro a añadir</param>
    /// <exception cref="GestionaApiException">En caso de que no pueda añadir el registro al expediente</exception>
    public async Task AddToFile(FileModel fileModel, RegistryAnnotationModel registryAnnotationModel)
    {
        try
        {
            var fileReference = GestionaApiUtilsService.ObtainLinkByRel(fileModel.Links!, GestionaApiUtilsService.SELF_REL);
            var url = GestionaApiUtilsService.ObtainLinkByRel(registryAnnotationModel.Links!, SEND_TO_FILE_REL);

            var linksModel = new LinksModel(FILE_REL, fileReference);

            var extraHeaders = new Dictionary<string, string> { { GestionaApiUtilsService.CONTENT_TYPE, MEDIA_TYPE_LINKS } };

            var response = await gestionaApiUtilsService.CreatePetitionAsync<object>(url, HttpMethod.Post, extraHeaders, linksModel);

            if (!response.StatusCode.Equals(HttpStatusCode.OK))
            {
                throw new GestionaApiException($"No se ha podido añadir el registro {registryAnnotationModel.Code} al expediente {fileModel.Code}");
            }
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"Error al añadir el registro de entrada al expediente: {e}");
        }
    }

    /// <summary>
    /// Obtiene la anotación de Gestiona a partir de la url que apunta a su recurso
    /// </summary>
    /// <param name="inputRegistryLocation">Link al recurso del registro buscado en Gestiona</param>
    /// <returns>Los datos de la anotación buscada</returns>
    public async Task<RegistryAnnotationModel> ObtainRegistryAnnotationModel(string inputRegistryLocation)
    {
        var response = await gestionaApiUtilsService.CreatePetitionAsync<RegistryAnnotationModel>(
            inputRegistryLocation,
            HttpMethod.Get,
            null,
            null);

        return response.Body!;
    }

    /// <summary>
    /// Finaliza el registro de entrada
    /// </summary>
    /// <param name="registryAnnotationModel">Datos del registro de entrada a finalizar</param>
    /// <exception cref="GestionaApiException">En caso de que no pueda finalizar el registro</exception>
    public async Task FinalizeRegistryAnnotation(RegistryAnnotationModel registryAnnotationModel)
    {
        try
        {
            var url = GestionaApiUtilsService.ObtainLinkByRel(registryAnnotationModel.Links!, FINALIZE_REL);

            var response = await gestionaApiUtilsService.CreatePetitionAsync<object>(url, HttpMethod.Post, null, null);

            if (!response.StatusCode.Equals(HttpStatusCode.OK))
            {
                throw new GestionaApiException($"No se ha podido finalizar el registro {registryAnnotationModel.Code}");
            }
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"Error al finalizar el registro {registryAnnotationModel.Code}: {e}");
        }
    }

    /// <summary>
    /// Obtiene los datos del documento tramitado a través de los datos de la anotación a la que está asociado
    /// </summary>
    /// <param name="registryAnnotationModel">Datos de la anotación a la que está asociado el documento</param>
    /// <returns>Datos del documento tramitado</returns>
    /// <exception cref="GestionaApiException">En caso de que no pueda obtener el documento tramitado</exception>
    public async Task<AnnotationDocumentModel> GetProcessedDocumentModel(RegistryAnnotationModel registryAnnotationModel)
    {
        try
        {
            var url = GestionaApiUtilsService.ObtainLinkByRel(registryAnnotationModel.Links!, DOCUMENTS_REL);

            var response = await gestionaApiUtilsService.CreatePetitionAsync<AnnotationDocumentPageModel>(url, HttpMethod.Get, null, null);
            return response.Body!.Content!.FirstOrDefault(document => !document.Receipt) ??
                   throw new GestionaApiException("No se ha encontrado ningún documento tramitado");
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"No se han podido obtener los datos del documento procesado: {e}");
        }
    }

    /// <summary>
    /// Obtiene el listado de oficinas de registro existentes en Gestiona
    /// </summary>
    /// <returns>El listado de las oficinas de registro</returns>
    /// <exception cref="GestionaApiException">En caso de que no haya podido obtener el listado</exception>
    private async Task<List<RegistryOfficeModel>> GetRegistryOfficeModels()
    {
        var url = gestionaApiUtilsService.GetBookmarkUrl(REGISTRY_OFFICES_REL);

        return (await gestionaApiUtilsService.CreatePetitionAsync<RegistryOfficesListModel>(url, HttpMethod.Get, null, null)).Body!.Content
               ?? throw new GestionaApiException("No se ha podido obtener el listado de expedientes");
    }

    /// <summary>
    /// Obtiene el link al recurso de la oficina de registro cuyo código coincide con el definido en el AppSettings
    /// </summary>
    /// <param name="registryOfficeModelList">Listado de oficinas de registro</param>
    /// <returns>El link a la oficina de registro buscada</returns>
    /// <exception cref="GestionaApiException">En caso de que no encuentre la oficina de registro</exception>
    private string ObtainInputRegistryOfficeLink(List<RegistryOfficeModel> registryOfficeModelList)
    {
        if (registryOfficeModelList == null || registryOfficeModelList.Count == 0)
        {
            throw new GestionaApiException("La lista de oficinas de registro está vacía o es nula.");
        }

        var registryOffice = registryOfficeModelList
                                 .FirstOrDefault(office => office.Code == this._registryOfficeCode)
                             ?? throw new GestionaApiException("Oficina de registro no encontrada.");

        var inputLink = registryOffice.Links!
                            .FirstOrDefault(link => link.Rel!.Equals(INPUT_LINK, StringComparison.OrdinalIgnoreCase))
                        ?? throw new GestionaApiException("Enlace 'input' no encontrado.");

        return inputLink.Href!;
    }

}