namespace mockup_gestiona_for_developers_net.Gestiona.ApiRest.Services;

using Exceptions;
using Models;
using Exception = System.Exception;

public class ProcedureService(GestionaApiUtilsService gestionaApiUtilsService)
{
    private const string CREATE_FILE = "create-file";
    private const string VND_GESTIONA_CATALOG = "vnd.gestiona.catalog-2015";

    /// <summary>
    /// Obtiene el procedimiento del catálogo nuevo a partir de su nombre
    /// </summary>
    /// <param name="procedureName">Nombre del procedimiento a buscar</param>
    /// <returns>Los datos del procedimiento</returns>
    /// <exception cref="GestionaApiException">En caso de que no sea capaz de encontrarlo</exception>
    public async Task<ProcedureModel> GetProcedure(string procedureName)
    {
        try
        {
            var url = gestionaApiUtilsService.GetBookmarkUrl(VND_GESTIONA_CATALOG);
            var response = await gestionaApiUtilsService.CreatePetitionAsync<ProcedurePageModel>(url, HttpMethod.Get, null, null);

            return await SearchProcedureRecursive(response.Body!, procedureName);
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"Error al obtener el procedimiento {procedureName}: {e}");
        }
    }

    /// <summary>
    /// Obtiene el trámite externo a partir de su título
    /// </summary>
    /// <param name="url">Recurso del procedimiento donde se encuentran los trámites externos</param>
    /// <param name="externalProcedureTitle">Título del trámite externo a buscar</param>
    /// <returns>Los datos del trámite externo</returns>
    /// <exception cref="GestionaApiException">En caso de que no encuentre el trámite externo</exception>
    public async Task<ExternalProcedureModel> GetExternalProcedure(string url, string externalProcedureTitle)
    {
        try
        {
            var response = await gestionaApiUtilsService.CreatePetitionAsync<ExternalProcedurePageModel>(url, HttpMethod.Get, null, null);

            return await SearchExternalProcedureRecursive(response.Body!, externalProcedureTitle);
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"Error al obtener el procedimiento {externalProcedureTitle}: {e}");
        }
    }

    /// <summary>
    /// Crea un nuevo expediente a partir de un trámite externo concreto
    /// </summary>
    /// <param name="externalProcedureModel">El trámite externo con el que se va a crear el nuevo expediente</param>
    /// <returns>Los datos de la apertura del expediente que se han generado</returns>
    /// <exception cref="GestionaApiException">En caso de que no sea capaz de crear el expediente</exception>
    public async Task<FileOpeningModel> CreateFile(ExternalProcedureModel externalProcedureModel)
    {
        try
        {
            var url = GestionaApiUtilsService.ObtainLinkByRel(externalProcedureModel.Links!, CREATE_FILE);
            var response = await gestionaApiUtilsService.CreatePetitionAsync<FileOpeningModel>(url, HttpMethod.Post, null, null);

            return response.Body!;
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"Error al crear el archivo para el trámite externo {externalProcedureModel.Title}: {e}");
        }
    }

    /// <summary>
    /// Función recursiva para buscar el procedimiento en las páginas
    /// </summary>
    /// <param name="procedurePageModel">Página en la que se está buscando</param>
    /// <param name="procedureName">Nombre del procedimiento buscado</param>
    /// <returns>Los datos del procedimiento si lo encuentra</returns>
    /// <exception cref="GestionaApiException">En caso de que no encuentre el procedimiento</exception>
    private async Task<ProcedureModel> SearchProcedureRecursive(ProcedurePageModel procedurePageModel, string procedureName)
    {
        try
        {
            var procedureModel = procedurePageModel.Content!.FirstOrDefault(procedure => procedureName.Equals(procedure.Name, StringComparison.Ordinal));

            // Caso base 1: encuentra el procedimiento en la página actual
            if (procedureModel != null)
            {
                return procedureModel;
            }

            // Caso recursivo: sigue a la siguiente página hasta encontrar el procedimiento o llegar a una página vacía
            if (procedurePageModel.Links!.Any(link => link.Rel == GestionaApiUtilsService.NEXT_REL))
            {
                var nextPageProcedureModel = await gestionaApiUtilsService.GetNextPageAsync<ProcedurePageModel, ProcedureModel>(procedurePageModel);

                return await SearchProcedureRecursive(nextPageProcedureModel, procedureName);
            }

            // Caso base 2: no encuentra el procedimiento en ninguna página
            throw new GestionaApiException($"No se ha encontrado el procedimiento {procedureName}");
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"Error al obtener el procedimiento {procedureName}: {e}");
        }
    }

    /// <summary>
    /// Función para buscar recursivamente el trámite externo en las páginas
    /// </summary>
    /// <param name="externalProcedurePageModel">Página en la que se está buscando</param>
    /// <param name="externalProcedureTitle">Título del trámite externo buscado</param>
    /// <returns>Los datos del trámite externo si lo encuentra</returns>
    /// <exception cref="GestionaApiException">En caso de que no encuentre el trámite externo</exception>
    private async Task<ExternalProcedureModel> SearchExternalProcedureRecursive(ExternalProcedurePageModel externalProcedurePageModel,
        string externalProcedureTitle)
    {
        try
        {
            var externalProcedureModel = externalProcedurePageModel.Content!.FirstOrDefault(procedure => externalProcedureTitle.Equals(procedure.Title, StringComparison.Ordinal));

            // Caso base 1: encuentra el procedimiento en la página actual
            if (externalProcedureModel != null)
            {
                return externalProcedureModel;
            }

            // Caso recursivo: sigue a la siguiente página hasta encontrar el procedimiento o llegar a una página vacía
            if (externalProcedurePageModel.Links!.Any(link => link.Rel == GestionaApiUtilsService.NEXT_REL))
            {
                var nextExternalProcedurePage =
                    await gestionaApiUtilsService.GetNextPageAsync<ExternalProcedurePageModel, ExternalProcedureModel>(externalProcedurePageModel);

                return await SearchExternalProcedureRecursive(nextExternalProcedurePage, externalProcedureTitle);
            }

            // Caso base 2: no encuentra el procedimiento en ninguna página
            throw new GestionaApiException($"No se ha encontrado el procedimiento {externalProcedureTitle}");
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"Error al obtener el procedimiento {externalProcedureTitle}: {e}");
        }
    }
}