namespace mockup_gestiona_for_developers_net.Gestiona.ApiRest.Services;

using System.Collections;
using System.Net;
using Exceptions;
using Google.Apis.Logging;
using Models;
using Exception = System.Exception;

public class DocumentService(GestionaApiUtilsService gestionaApiUtilsService)
{

    private const string MEDIA_TYPE_FILE_DOCUMENT = "application/vnd.gestiona.file-document+json;version=3";
    private const string MEDIA_TYPE_CIRCUIT_TEMPLATE = "application/vnd.gestiona.circuits.template-filedoc+json;version=5";
    private const string CONTENT_REL = "content";
    private const string CIRCUIT_REL = "circuit";
    private const string DOCUMENT_ANNOTATIONS_REL = "document-annotations";

    /// <summary>
    /// Obtiene el documento a partir de la url que le apunta en Gestiona
    /// </summary>
    /// <param name="documentReference">Link al recurso del documento en Gestiona</param>
    /// <returns>Los datos del documento obtenidos</returns>
    /// <exception cref="GestionaApiException">En caso de que no pueda obtener el documento</exception>
    public async Task<DocumentModel> ObtainDocument(string documentReference)
    {
        try
        {
            var response = await gestionaApiUtilsService.CreatePetitionAsync<DocumentModel>(documentReference, HttpMethod.Get, null, null);

            return response.Body!;
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"No se ha podido obtener el documento {e}");
        }
    }

    /// <summary>
    /// Obtiene los circuitos de tramitación disponible para un documento a través de su recurso a los circuitos de tramitación
    /// </summary>
    /// <param name="circuitTemplatesReference">Url sobre la cuál realizar la petición para obtener los circuitos</param>
    /// <returns>El listado de los circuitos de tramitación disponibles</returns>
    /// <exception cref="GestionaApiException">En el caso de que no pueda obtener los circuitos de tramitación</exception>
    public async Task<List<CircuitTemplateModel>> ObtainCircuitTemplates(string circuitTemplatesReference)
    {
        try
        {
            var response = await gestionaApiUtilsService.CreatePetitionAsync<CircuitTemplatePageModel>(circuitTemplatesReference, HttpMethod.Get, null, null);

            return response.Body!.Content!;
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"No se han podido obtener los circuitos de tramitación: {e}");
        }
    }

    /// <summary>
    /// Crea el documento en Gestiona 
    /// </summary>
    /// <param name="documentsAndFoldersUrl">Url sobre la que hacer la petición</param>
    /// <param name="uploadUrl">Url al recurso upload donde se ha subido el contenido del fichero</param>
    /// <param name="fileName">Nombre del documento a subir</param>
    /// <returns>La url al recurso donde se ha creado el documento</returns>
    /// <exception cref="GestionaApiException">En el caso de que no haya podido crear el documento</exception>
    public async Task<string> CreateDocument(string documentsAndFoldersUrl, string uploadUrl, string fileName)
    {
        try
        {
            var links = new List<LinkModel> { new(CONTENT_REL, uploadUrl) };

            var newDocument = new DocumentModel(links, fileName);

            var extraHeaders = new Dictionary<string, string> { { GestionaApiUtilsService.CONTENT_TYPE, MEDIA_TYPE_FILE_DOCUMENT } };

            var response =
                await gestionaApiUtilsService.CreatePetitionAsync<DocumentModel>(documentsAndFoldersUrl, HttpMethod.Post, extraHeaders, newDocument);

            return GestionaApiUtilsService.ObtainLinkByRel(response.Body!.Links!, GestionaApiUtilsService.SELF_REL);
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"No se ha podido crear el documento: {e}");
        }
    }

    /// <summary>
    /// Tramita el documento con el circuito de tramitación especificado
    /// </summary>
    /// <param name="documentReference">Referencia al documento a tramitar</param>
    /// <param name="circuitTemplateModel">Datos del circuito de tramitación a aplicar</param>
    /// <exception cref="GestionaApiException">En caso de que no pueda tramitar el documento</exception>
    public async Task ProcessDocument(string documentReference, CircuitTemplateModel circuitTemplateModel)
    {
        var documentModel = await ObtainDocument(documentReference);
        var processDocumentUrl = GestionaApiUtilsService.ObtainLinkByRel(documentModel.Links!, CIRCUIT_REL);

        try
        {
            var extraHeaders = new Dictionary<string, string> { { GestionaApiUtilsService.CONTENT_TYPE, MEDIA_TYPE_CIRCUIT_TEMPLATE } };

            var response = await gestionaApiUtilsService.CreatePetitionAsync<object>(processDocumentUrl, HttpMethod.Post, extraHeaders, circuitTemplateModel);

            if (response.StatusCode != HttpStatusCode.Created)
            {
                throw new GestionaApiException($"No se ha podido tramitar el documento: {response.StatusCode}");
            }
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"Error al tramitar el documento: {e}");
        }
    }

    /// <summary>
    /// Comprueba el estado de tramitación de un documento
    /// </summary>
    /// <param name="documentReference">Referencia al documento en Gestiona</param>
    /// <returns>El estado de la tramitación del documento</returns>
    /// <exception cref="GestionaApiException">En caso de que no pueda obtener el estado del trámite</exception>
    public async Task<DocumentProcessStateModel> CheckDocumentProcessStatus(string documentReference)
    {
        var documentModel = await ObtainDocument(documentReference);
        var processDocumentUrl = GestionaApiUtilsService.ObtainLinkByRel(documentModel.Links!, CIRCUIT_REL);

        try
        {
            var response = await gestionaApiUtilsService.CreatePetitionAsync<DocumentProcessStateModel>(processDocumentUrl, HttpMethod.Get, null, null);

            return response.Body!;
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"No se ha podido obtener el estado del trámite del documento: {e}");
        }
    }

    /// <summary>
    /// Obtiene el listado de los datos de los registros asociados a un documento a partir de la referencia del documento en Gestiona
    /// </summary>
    /// <param name="documentReference">Referencia al documento en Gestiona</param>
    /// <returns>El listado de los datos de los registros asociados a ese documento</returns>
    /// <exception cref="GestionaApiException">En caso de que no pueda obtener el listado</exception>
    public async Task<List<FileDocumentAnnotationModel>?> GetDocumentAnnotations(string documentReference)
    {
        try
        {
            var documentModel = await ObtainDocument(documentReference);
            var annotationsUrl = GestionaApiUtilsService.ObtainLinkByRel(documentModel.Links!, DOCUMENT_ANNOTATIONS_REL);

            var response = await gestionaApiUtilsService.CreatePetitionAsync<FileDocumentAnnotationPageModel>(annotationsUrl, HttpMethod.Get, null, null);

            return response.Body?.Content;
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"No se han podido obtener las anotaciones del documento con referencia {documentReference}: {e}");
        }
    }

    /// <summary>
    /// Obtiene el contenido de un documento tramitado a partir de los datos del documento
    /// </summary>
    /// <param name="documentModel">Datos del documento</param>
    /// <returns>Los bytes del contenido del documento</returns>
    /// <exception cref="GestionaApiException">En caso de que no pueda obtener el contenido</exception>
    public async Task<byte[]> GetAnnotationDocumentContent(DocumentModel documentModel)
    {
        try
        {
            var url = GestionaApiUtilsService.ObtainLinkByRel(documentModel.Links!, CONTENT_REL);

            var response = await gestionaApiUtilsService.CreateDocumentPetitionAsync(url);

            return response;
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"No se ha podido obtener el contenido del documento: {e}");
        }
    }
}