using System.Net.Http.Headers;
using System.Text.Json;
using mockup_gestiona_for_developers_net.Gestiona.ApiRest.Exceptions;
using mockup_gestiona_for_developers_net.Gestiona.ApiRest.Models;

namespace mockup_gestiona_for_developers_net.Gestiona.ApiRest;

using System.Globalization;
using System.Text;
using System.Text.Json.Serialization;
using Configuration;
using Exception = System.Exception;

public class GestionaApiUtilsService(MockupSettings mockupSettings, ILogger<GestionaApiUtilsService> logger) : IDisposable
{
    public const string CONTENT_TYPE = "Content-Type";
    public const string SELF_REL = "self";
    public const string NEXT_REL = "next";
    public static readonly JsonSerializerOptions SerializerOptions = new() { PropertyNameCaseInsensitive = true };

    private const string X_GESTIONA_ACCESS_TOKEN = "X-Gestiona-Access-Token";
    private const string APPLICATION_OCTET_STREAM = "application/octet-stream";
    private const string VERSION_PARAMETER = "version";

    private static readonly Action<ILogger, string, Exception?> LogErrorReadingBookmarksAction =
        LoggerMessage.Define<string>(
            LogLevel.Error,
            new EventId(1008, nameof(LogErrorReadingBookmarks)),
            "{Message}");

    private List<LinkModel> _bookmarks = [];
    private readonly CompositeFormat _formatBookmarks = CompositeFormat.Parse("{0}/rest");
    private readonly HttpClient _httpClient = new();
    private readonly string? _gestionaApiUrl = mockupSettings.GestionaApiSettings.Url;
    private readonly string? _accessToken = mockupSettings.GestionaApiSettings.AccessToken;

    /// <summary>
    /// Obtiene el link de de un LinkModel a través de su relación
    /// </summary>
    /// <param name="linkModelList">Lista de links en los que buscar</param>
    /// <param name="rel">Nombre de la relación buscada</param>
    /// <returns>La url en el caso de que lo encuentre</returns>
    /// <exception cref="GestionaApiException">En el caso de que no se encuentre el link</exception>
    public static string ObtainLinkByRel(List<LinkModel> linkModelList, string rel)
    {
        var link = linkModelList
            .FirstOrDefault(link => string.Equals(link.Rel, rel, StringComparison.OrdinalIgnoreCase));

        if (link?.Href == null)
        {
            throw new GestionaApiException($"No se ha encontrado un link con la relación: {rel}");
        }

        return link.Href;
    }

    /// <summary>
    /// Método genérico para crear una petición HTTP
    /// </summary>
    /// <param name="uri">Url sobre la cuál se va a generar la petición</param>
    /// <param name="method">Método de la petición a generar</param>
    /// <param name="extraHeaders">Cabeceras extras</param>
    /// <param name="body">Cuerpo de la petición</param>
    /// <typeparam name="T">Tipo de dato que se va a recibir en la petición</typeparam>
    /// <returns>El resultado de la petición generada</returns>
    public async Task<ApiResponseModel<T>> CreatePetitionAsync<T>(string uri, HttpMethod method, Dictionary<string, string>? extraHeaders, object? body)
    {
        using var request = new HttpRequestMessage(method, uri);
        request.Headers.Add(X_GESTIONA_ACCESS_TOKEN, this._accessToken);

        if (body != null)
        {
            request.Content = new StringContent(JsonSerializer.Serialize(body));
            if (extraHeaders != null)
            {
                CreateContentTypeHeader(extraHeaders[CONTENT_TYPE], request);
            }
        }

        if (extraHeaders != null)
        {
            foreach (var header in extraHeaders.Where(header => !header.Key.Equals(CONTENT_TYPE, StringComparison.OrdinalIgnoreCase)))
            {
                request.Headers.Add(header.Key, header.Value);
            }
        }

        using var response = await this._httpClient.SendAsync(request);
        return await MapToApiResponseModel<T>(response);
    }

    /// <summary>
    /// Realiza una petición sobre la API de Gestiona para subir el contenido de un documento
    /// </summary>
    /// <param name="uri">URI del recurso sobre el que realizar la petición</param>
    /// <param name="content">Contenido binario del documento a subir</param>
    /// <returns>La respuesta a la petición realizada</returns>
    public async Task<HttpResponseMessage> UploadDocumentPetitionAsync(string uri, ByteArrayContent content)
    {
        using var request = new HttpRequestMessage(HttpMethod.Put, uri);
        content.Headers.ContentType = MediaTypeHeaderValue.Parse(APPLICATION_OCTET_STREAM);
        request.Headers.Add(X_GESTIONA_ACCESS_TOKEN, this._accessToken);
        request.Content = content;

        return await this._httpClient.SendAsync(request);
    }

    /// <summary>
    /// Crea una petición para obtener el contenido de un documento
    /// </summary>
    /// <param name="uri">Url sobre la que realizar la petición</param>
    /// <returns>El contenido del documento en bytes</returns>
    /// <exception cref="GestionaApiException">En caso dee que no pueda obtener el contenido del documento</exception>
    public async Task<byte[]> CreateDocumentPetitionAsync(string uri)
    {
        using var request = new HttpRequestMessage(HttpMethod.Get, uri);
        request.Headers.Add(X_GESTIONA_ACCESS_TOKEN, this._accessToken);

        using var response = await this._httpClient.SendAsync(request);

        if (!response.IsSuccessStatusCode)
        {
            throw new GestionaApiException($"Error al obtener el contenido del documento: {response.StatusCode}");
        }

        return await response.Content.ReadAsByteArrayAsync();
    }

    /// <summary>
    /// Obtiene la siguiente página de un modelo de página genérico
    /// </summary>
    /// <typeparam name="TPageModel">El tipo de la página (que hereda de PageModel)</typeparam>
    /// <typeparam name="TContent">Tipo de dato del contenido de la página</typeparam>
    /// <param name="pageModel">Página en la que se encuentra</param>
    /// <returns>La siguiente página si existe</returns>
    /// <exception cref="GestionaApiException">En caso de que no pueda obtener la siguiente página</exception>
    public async Task<TPageModel> GetNextPageAsync<TPageModel, TContent>(TPageModel pageModel) where TPageModel : PageModel<TContent>, new()
    {
        var nextPageUrl = ObtainLinkByRel(pageModel.Links!, NEXT_REL);

        if (string.IsNullOrEmpty(nextPageUrl))
        {
            throw new GestionaApiException($"No tiene una siguiente página");
        }

        try
        {
            var response = await CreatePetitionAsync<TPageModel>(nextPageUrl, HttpMethod.Get, null, null);
            return response.Body!;
        }
        catch (Exception ex)
        {
            throw new GestionaApiException($"Error al obtener la siguiente página: {ex}");
        }
    }

    /// <summary>
    /// Obtiene el bookmark especificado
    /// </summary>
    /// <param name="linkRel">Nombre de la relación del bookmark a buscar</param>
    /// <returns>La URL del bookmark buscado en el caso de que lo encuentre</returns>
    /// <exception cref="GestionaApiException">En el caso de que no lo encuentre</exception>
    public string GetBookmarkUrl(string linkRel)
    {
        ReloadBookmarksIfNotPresent();
        var bookmark = this._bookmarks
            .FirstOrDefault(b => string.Equals(b.Rel, linkRel, StringComparison.OrdinalIgnoreCase));

        if (bookmark?.Href == null)
        {
            throw new GestionaApiException($"No se ha encontrado un bookmark con la relación: {linkRel}");
        }

        return bookmark.Href;
    }

    /// <summary>
    /// Inicializa el listado de bookmarks
    /// </summary>
    /// <exception cref="GestionaApiException">En el caso de que no los pueda obtener</exception>
    public async Task ObtainBookmarksAsync()
    {
        try
        {
            // Formatear la URL
            var url = string.Format(CultureInfo.InvariantCulture, this._formatBookmarks, this._gestionaApiUrl);

            var response = await CreatePetitionAsync<BookmarkModel>(
                url,
                HttpMethod.Get,
                null,
                null
            );

            if (response.Body?.Links == null)
            {
                this._bookmarks = [];
                LogErrorReadingBookmarks("No se han podido leer los bookmarks correctamente: Respuesta vacía o nula.");
                return;
            }

            this._bookmarks = response.Body.Links;
        }
        catch (Exception e)
        {
            throw new GestionaApiException($"No se han podido leer los bookmarks correctamente: {e.Message}");
        }
    }

    /// <summary>
    /// Mapea la respuesta a la API de Gestiona a un `ApiResponseModel`
    /// </summary>
    /// <param name="httpResponse">Respuesta a la petición de la API de Gestiona</param>
    /// <typeparam name="T">Tipo de dato esperado en el body de la respuesta</typeparam>
    /// <returns>La `ApiResponseModel` a la que se ha mapeado la respuesta</returns>
    private static async Task<ApiResponseModel<T>> MapToApiResponseModel<T>(HttpResponseMessage httpResponse)
    {
        T? body = default;
        {
            var bodyContent = await httpResponse.Content.ReadAsStringAsync();

            if (!string.IsNullOrWhiteSpace(bodyContent))
            {
                body = JsonSerializer.Deserialize<T>(bodyContent, SerializerOptions);
            }
        }

        return new ApiResponseModel<T>(body, httpResponse.Headers, httpResponse.StatusCode);
    }

    /// <summary>
    /// Recarga los bookmarks en el caso de que no estén correctamente inicializados
    /// </summary>
    private void ReloadBookmarksIfNotPresent()
    {
        if (this._bookmarks.Count.Equals(0))
        {
            _ = ObtainBookmarksAsync();
        }
    }

    /// <summary>
    /// Crea la cabecera para el Content-Type comprobando si este tiene el parámetro version
    /// </summary>
    /// <param name="contentType">Tipo de dato del contenido</param>
    /// <param name="request"></param>
    private static void CreateContentTypeHeader(string contentType, HttpRequestMessage request)
    {
        var splittedHeader = contentType.Split(';');

        MediaTypeHeaderValue mediaType;
        if (splittedHeader.Length > 1)
        {
            mediaType = new MediaTypeHeaderValue(splittedHeader[0]);
            var versionParameter = splittedHeader[1].Split("=");
            mediaType.Parameters.Add(new NameValueHeaderValue(VERSION_PARAMETER, versionParameter[1]));
        }
        else
        {
            mediaType = new MediaTypeHeaderValue(contentType);
        }

        request.Content!.Headers.ContentType = mediaType;
    }

    [JsonConverter(typeof(JsonStringEnumConverter))]
    public enum NotificationChannelType
    {
        [JsonPropertyName("TELEMATIC")] TELEMATIC,
        [JsonPropertyName("PAPER")] PAPER
    }

    public void Dispose()
    {
        this._httpClient.Dispose();
        GC.SuppressFinalize(this);
    }

    private void LogErrorReadingBookmarks(string message)
    {
        LogErrorReadingBookmarksAction(logger, message, null);
    }

}