using mockup_gestiona_for_developers_net.Gestiona.ApiRest.Models;
using mockup_gestiona_for_developers_net.Gestiona.ApiRest.Services;

namespace mockup_gestiona_for_developers_net.Gestiona.ApiRest;

using Exceptions;
using Exception = System.Exception;

public class GestionaApiService(
    FileService fileService,
    RegistryService registryService,
    DocumentService documentService,
    UploadService uploadService,
    GestionaApiUtilsService gestionaApiUtilsService,
    ProcedureService procedureService,
    ThirdService thirdService,
    ILogger<GestionaApiService> logger)
{

    private const string CIRCUIT_TEMPLATES_REL = "circuits-templates";
    private const string UPLOADS_BOOKMARK = "vnd.gestiona.uploads";
    private const string DOCUMENTS_AND_FOLDERS_REL = "documents-and-folders";
    private const string EXTERNAL_PROCEDURES_REL = "external-procedures";
    private const string DEFAULT_ADDRESS_REL = "default-address";
    private const string THIRDPARTIES_REL = "thirdparties";

    private static readonly Action<ILogger, string, Exception?> LogThirdExistsInGestionaAction =
        LoggerMessage.Define<string>(
            LogLevel.Information,
            new EventId(1006, nameof(LogThirdExistsInGestiona)),
            "{Message}");

    /// <summary>
    /// Obtiene el listado de expedientes de Gestiona
    /// </summary>
    /// <returns>El listado de expedientes</returns>
    public Task<List<FileModel>> GetFileList()
    {
        return fileService.GetFileList();
    }

    /// <summary>
    /// Crea un nuevo registro de entrada en Gestiona
    /// </summary>
    /// <param name="summary">Resumen del registro que se va a crear</param>
    /// <returns>Los datos del registro una vez creado en Gestiona</returns>
    public async Task<RegistryAnnotationModel> CreateInputRegistry(string summary)
    {
        return await registryService.CreateInputRegistry(summary);
    }

    /// <summary>
    /// Obtiene el listado de circuitos de tramitación disponibles para un documento
    /// </summary>
    /// <param name="documentReference">Url que apunta al recurso del documento a consultar en Gestiona</param>
    /// <returns>El listado de circuitos de tramitación</returns>
    public async Task<List<CircuitTemplateModel>> ObtainCircuitTemplates(string documentReference)
    {
        var documentModel = await documentService.ObtainDocument(documentReference);
        var circuitTemplateUrl = GestionaApiUtilsService.ObtainLinkByRel(documentModel.Links!, CIRCUIT_TEMPLATES_REL);
        return await documentService.ObtainCircuitTemplates(circuitTemplateUrl);
    }

    /// <summary>
    /// Sube un documento a Gestiona y lo crea para un expediente determinado
    /// </summary>
    /// <param name="fileModel">Expediente al que añadirle el documento</param>
    /// <param name="documentPath">Ruta del documento en local</param>
    /// <returns>La url al recurso del documento creado</returns>
    public async Task<string> AddDocumentToFile(FileModel fileModel, string documentPath)
    {
        var documentsAndFoldersUrl = GestionaApiUtilsService.ObtainLinkByRel(fileModel.Links!, DOCUMENTS_AND_FOLDERS_REL);

        var uploadBookmark = gestionaApiUtilsService.GetBookmarkUrl(UPLOADS_BOOKMARK);

        var uploadReference = await uploadService.CreateEmptyUpload(uploadBookmark);
        var fileName = await uploadService.UploadDocument(uploadReference, documentPath);

        return await documentService.CreateDocument(documentsAndFoldersUrl, uploadReference, fileName);
    }

    /// <summary>
    /// Tramita el documento con el circuito de tramitación especificado
    /// </summary>
    /// <param name="documentReference">Referencia al documento en Gestiona</param>
    /// <param name="circuitTemplateModel">Datos del circuito de tramitación a aplicar</param>
    public async Task ProcessDocument(string documentReference, CircuitTemplateModel circuitTemplateModel)
    {
        await documentService.ProcessDocument(documentReference, circuitTemplateModel);
    }

    /// <summary>
    /// Comprueba el estado de la tramitación de un documento
    /// </summary>
    /// <param name="documentReference">Referencia al documento en Gestiona</param>
    /// <returns>Los datos del estado del trámite del documento</returns>
    public async Task<DocumentProcessStateModel> CheckDocumentProcessStatus(string documentReference)
    {
        return await documentService.CheckDocumentProcessStatus(documentReference);
    }

    /// <summary>
    /// Crea un nuevo expediente a partir de un procedimiento y un trámite externo
    /// </summary>
    /// <param name="procedureName">Nombre del procedimiento</param>
    /// <param name="externalProcedureTitle">Título del trámite externo</param>
    /// <returns>Los datos del expediente que se ha creado</returns>
    public async Task<FileModel> CreateFileWithProcedure(string procedureName, string externalProcedureTitle)
    {
        var procedureModel = await procedureService.GetProcedure(procedureName);

        var externalProcedureModel = await procedureService.GetExternalProcedure(
            GestionaApiUtilsService.ObtainLinkByRel(procedureModel.Links!, EXTERNAL_PROCEDURES_REL),
            externalProcedureTitle);

        var fileOpeningModel = await procedureService.CreateFile(externalProcedureModel);

        return await fileService.OpenFile(fileOpeningModel);
    }

    /// <summary>
    /// Añade un registro de entrada a un expediente
    /// </summary>
    /// <param name="fileModel">Datos del expediente</param>
    /// <param name="registryAnnotationModel">Datos del registro de entrada que se va a añadir al expediente</param>
    public async Task AddRegistryToFile(FileModel fileModel, RegistryAnnotationModel registryAnnotationModel)
    {
        await registryService.AddToFile(fileModel, registryAnnotationModel);
    }

    /// <summary>
    /// Obtiene los datos de un registro
    /// </summary>
    /// <param name="registryAnnotationModel">Datos del registro en Gestiona</param>
    /// <returns>Los datos del registro de Gestiona</returns>
    public async Task<RegistryAnnotationModel> ObtainRegistryAnnotation(RegistryAnnotationModel registryAnnotationModel)
    {
        var registryReference = GestionaApiUtilsService.ObtainLinkByRel(registryAnnotationModel.Links!, GestionaApiUtilsService.SELF_REL);
        return await registryService.ObtainRegistryAnnotationModel(registryReference);
    }

    /// <summary>
    /// Obtiene los datos de la anotación asociada al documento especificado
    /// </summary>
    /// <param name="documentReference">Referencia al documento buscado</param>
    /// <returns>Los datos de la anotación asociada en el caso de que tenga uno, null en caso contrario</returns>
    public async Task<FileDocumentAnnotationModel?> GetFileDocumentAnnotationModel(string documentReference)
    {
        var fileDocumentAnnotationList = await documentService.GetDocumentAnnotations(documentReference);

        if (fileDocumentAnnotationList == null || fileDocumentAnnotationList.Count == 0)
        {
            return null;
        }

        return fileDocumentAnnotationList.First();
    }

    /// <summary>
    /// Obtiene los datos del documento a partir de la referencia a su recurso en Gestiona
    /// </summary>
    /// <param name="documentReference">Referencia al recurso del documento en Gestiona</param>
    /// <returns>Datos del documento</returns>
    public async Task<DocumentModel> GetDocument(string documentReference)
    {
        return await documentService.ObtainDocument(documentReference);
    }

    /// <summary>
    /// Obtiene los datos de salida de un tercero a través de su NIF
    /// </summary>
    /// <param name="nif">Nif del tercero a buscar en Gestiona</param>
    /// <returns>Los datos de salida del tercero</returns>
    public async Task<ThirdOutputModel> GetThirdOutputByNif(string nif)
    {
        var thirdModel = await thirdService.GetThirdByNif(nif);

        var thirdOutputModel = ThirdService.CreateThirdOutputModel(thirdModel);

        if (thirdModel.NotificationChannel.Equals(GestionaApiUtilsService.NotificationChannelType.PAPER))
        {
            var thirdAddress = await thirdService.GetThirdAddressModel(GestionaApiUtilsService.ObtainLinkByRel(thirdModel.Links!, DEFAULT_ADDRESS_REL));
            ThirdService.AddPaperNecessaryFields(thirdOutputModel, thirdAddress);
        }

        return thirdOutputModel;
    }

    /// <summary>
    /// Obtiene los datos del tercero asociado a través de su Nif
    /// </summary>
    /// <param name="nif">Nif del tercero a buscar</param>
    /// <returns>Los datos del tercero asociado</returns>
    public async Task<ThirdPartyModel> GetThirdPartyByNif(string nif)
    {
        var thirdModel = await thirdService.GetThirdByNif(nif);
        return ThirdService.CreateThirdPartyModel(thirdModel);
    }

    /// <summary>
    /// Añade el tercero a un registro de entrada
    /// </summary>
    /// <param name="registryAnnotationModel">Datos del registro de entrada al que se va a añadir</param>
    /// <param name="thirdPartyModel">Datos del tercero asociado a añadir</param>
    public async Task AddThirdPartyToInputRegistry(RegistryAnnotationModel registryAnnotationModel, ThirdPartyModel thirdPartyModel)
    {
        var registryUrl = GestionaApiUtilsService.ObtainLinkByRel(registryAnnotationModel.Links!, THIRDPARTIES_REL);
        await thirdService.AddThirdPartyToInputRegistry(registryUrl, thirdPartyModel);
    }

    /// <summary>
    /// Finaliza un registro de entrada
    /// </summary>
    /// <param name="registryAnnotationModel">Registro de entrada a finalizar</param>
    public async Task FinalizeRegistryAnnotation(RegistryAnnotationModel registryAnnotationModel)
    {
        await registryService.FinalizeRegistryAnnotation(registryAnnotationModel);
    }

    /// <summary>
    /// Obtiene el contenido de un documento tramitado a partir de los datos del documento
    /// </summary>
    /// <param name="documentModel">Datos del documento</param>
    /// <returns>Los bytes del contenido del documento</returns>
    public async Task<byte[]> GetDocumentContent(DocumentModel documentModel)
    {
        return await documentService.GetAnnotationDocumentContent(documentModel);
    }

    /// <summary>
    /// Comprueba si un tercero existe en Gestiona y en caso contrario lo crea
    /// </summary>
    /// <param name="nif">Nif del tercero a buscar</param>
    public async Task CheckIfThirdExists(string nif)
    {
        try
        {
            _ = await thirdService.GetThirdByNif(nif);
            LogThirdExistsInGestiona($"El tercero con dni {nif} existe en Gestiona");
        }
        catch (GestionaApiException)
        {
            LogThirdExistsInGestiona($"Se va a crear el tercero con DNI {nif} en Gestiona");
            await thirdService.CreateThird(nif);
        }
    }

    private void LogThirdExistsInGestiona(string message)
    {
        LogThirdExistsInGestionaAction(logger, message, null);
    }

}