using Google.Apis.Auth.OAuth2;
using Google.Apis.Services;
using Google.Apis.Sheets.v4;

namespace mockup_gestiona_for_developers_net.Connectors.ThirdData.Services;

using Configuration;
using Microsoft.Extensions.Options;
using Exception = System.Exception;

public class GoogleSheetsService(MockupSettings mockupSettings, ILogger<GoogleSheetsService> logger)
{
    private const string APPLICATION_NAME = "MockupForDevelopers";
    private const string RANGE = "Persona!A2:L";
    private const int NIF_COLUMN = 1;
    private const int INCIDENCE_NUMBER_COLUMN = 0;
    private const int INCIDENCE_GROUP_COLUMN = 11;

    private static readonly Action<ILogger, Exception?> LogNoValuesInSheetAction =
        LoggerMessage.Define(
            LogLevel.Warning,
            new EventId(1002, nameof(LogNoValuesInSheet)),
            "No se encontraron valores en la hoja de cálculo.");

    private static readonly Action<ILogger, string, string, Exception?> LogErrorFindingThirdAction =
        LoggerMessage.Define<string, string>(
            LogLevel.Error,
            new EventId(1003, nameof(LogErrorFindingThird)),
            "Error al buscar al tercero con DNI {Nif}: {Message}");

    private readonly string? _credentialsPath = mockupSettings.ConnectorsSettings.ThirdDataSettings.GoogleSheetsSettings.CredentialsPath;

    private readonly string? _spreadsheetId = mockupSettings.ConnectorsSettings.ThirdDataSettings.GoogleSheetsSettings.SpreadsheetId;

    private readonly SearchType? _searchType = mockupSettings.ConnectorsSettings.ThirdDataSettings.GoogleSheetsSettings.SearchType;

    /// <summary>
    /// Establece el servicio con la API de Google Sheets
    /// </summary>
    /// <returns>El servicio obtenido</returns>
    private SheetsService GetSheetsService()
    {
        GoogleCredential credential;
        using (var stream = new FileStream(this._credentialsPath!, FileMode.Open, FileAccess.Read))
        {
            credential = GoogleCredential.FromStream(stream)
                .CreateScoped(SheetsService.Scope.SpreadsheetsReadonly);
        }

        return new SheetsService(new BaseClientService.Initializer { HttpClientInitializer = credential, ApplicationName = APPLICATION_NAME });
    }

    /// <summary>
    /// Busca en la hoja de Google Sheets el tercero por su nif y devuelve sus datos en caso de que lo encuentre
    /// </summary>
    /// <param name="nif">Dni del tercero a buscar</param>
    /// <returns>El listado de datos del tercero en el caso de que lo haya encontrado o null en el caso contrario</returns>
    public async Task<List<object>?> GetThirdByNifAsync(string? nif)
    {
        try
        {
            var values = await GetAllThirds();

            if (values != null && values.Count != 0) return SearchThird(nif, values, NIF_COLUMN);
            LogNoValuesInSheet();
            return null;
        }
        catch (Exception ex)
        {
            LogErrorFindingThird(nif!, ex.Message, ex);
            return null;
        }
    }

    /// <summary>
    /// Realiza la búsqueda del tercero a través de la columna definida a nivel de configuración
    /// </summary>
    /// <param name="searchedValue">Valor del tercero de la columna definida a buscar</param>
    /// <returns>El listado de datos del tercero en el caso de que lo haya encontrado o null en el caso contrario</returns>
    public async Task<List<object>?> ResolveThirdSearch(string searchedValue)
    {
        var values = await GetAllThirds();

        if (values != null && values.Count != 0)
        {
            return this._searchType switch
            {
                SearchType.NIF => SearchThird(searchedValue, values, NIF_COLUMN),
                SearchType.INCIDENCE_NUMBER => SearchThird(searchedValue, values, INCIDENCE_NUMBER_COLUMN),
                SearchType.INCIDENCE_GROUP => SearchThird(searchedValue, values, INCIDENCE_GROUP_COLUMN),
                var _ => null
            };
        }

        return null;
    }

    /// <summary>
    /// Obtiene todos los datos de los terceros presentes en la hoja de Google Sheets
    /// </summary>
    /// <returns>La matriz con todos los datos obtenidos</returns>
    public async Task<IList<IList<object>>?> GetAllThirds()
    {
        try
        {
            var service = GetSheetsService();
            var request = service.Spreadsheets.Values.Get(this._spreadsheetId, RANGE);
            var response = await request.ExecuteAsync();
            var values = response.Values;

            if (values != null && values.Count != 0) return values;
            LogNoValuesInSheet();
            return null;
        }
        catch (Exception)
        {
            LogNoValuesInSheet();
            return null;
        }
    }

    /// <summary>
    /// Busca el tercero entre la matriz de valores que ha sacado de la hoja de Google Sheets
    /// </summary>
    /// <param name="searchedValue">Dni del tercero a buscar</param>
    /// <param name="values">Matriz de valores de la hoja de Google Sheets</param>
    /// <param name="column">Matriz de valores de la hoja de Google Sheets</param>
    /// <returns>El listado de datos del tercero en el caso de que lo haya encontrado o null en el caso contrario</returns>
    private List<object>? SearchThird(string? searchedValue, IList<IList<object>> values, int column)
    {
        foreach (var row in values)
        {
            if (row.Count > column && row[column].ToString() == searchedValue)
            {
                return row.ToList();
            }
        }

        LogNoValuesInSheet();
        return null;
    }

    private void LogNoValuesInSheet()
    {
        LogNoValuesInSheetAction(logger, null);
    }

    private void LogErrorFindingThird(string nif, string message, Exception exception)
    {
        LogErrorFindingThirdAction(logger, nif, message, exception);
    }

    public enum SearchType
    {
        INCIDENCE_GROUP,
        INCIDENCE_NUMBER,
        NIF
    }

}