using Microsoft.AspNetCore.Mvc;
using mockup_gestiona_for_developers_net.Connectors.Comm.Service;
using mockup_gestiona_for_developers_net.Connectors.Common.Models;
using mockup_gestiona_for_developers_net.Connectors.ThirdData.Services;

namespace mockup_gestiona_for_developers_net.Connectors.AssignedIncidence.Controller;


[ApiController]
[Route(THIRD_DATA_CONNECTOR_CONTROLLER_PREFIX)]
public class ThirdDataConnectorController(
    GoogleSheetsService googleSheetsService,
    ConnectorUtilsService connectorUtilsService)
{
    private const string THIRD_DATA_CONNECTOR_CONTROLLER_PREFIX = "connectors/third-data";
    private const string THIRD_MESSAGE_NOT_FOUND = "Third not found";

    /// <summary>
    /// Obtiene la versión actual de la API.
    /// </summary>
    /// <param name="token">El token de autenticación</param>
    /// <returns>La versión actual de la API.</returns>
    [HttpGet("versions/current")]
    public IActionResult GetCurrentVersion([FromHeader(Name = "X-Rest-Basic-Token")] string token)
    {
        // TODO: Validar el token

        return connectorUtilsService.GenerateVersionResponse(token);
    }

    /// <summary>
    /// Devuelve la lista de bookmarks
    /// </summary>
    /// <param name="token">El token JWT de autenticación</param>
    /// <returns>El listado de bookmarks del conector</returns>
    [HttpGet("")]
    public IActionResult GetBookmarks([FromHeader(Name = "X-Rest-Basic-Token")] string token)
    {
        // TODO: Validar el token

        var genericOperationUrl = connectorUtilsService.GetGenericOperationUrl(THIRD_DATA_CONNECTOR_CONTROLLER_PREFIX);

        return connectorUtilsService.GenerateBookmarksResponse(token, genericOperationUrl);
    }

    /// <summary>
    /// Recurso operación genérica del conector de datos de terceros. Busca el tercero en la hoja de Google Sheets y devuelve sus datos
    /// </summary>
    /// <param name="token">Token JWT de autenticación</param>
    /// <param name="requestData">Datos de la operación genérica. `FIELD_0`: dni del tercero</param>
    /// <returns>Los datos del tercero buscado o un error en el caso de que no lo encuentre o exista algún error con los datos recibidos</returns>
    [HttpPost("genericoperations")]
    public IActionResult GenericOperation([FromHeader(Name = "X-Rest-Basic-Token")] string token, [FromBody] DataModel requestData)
    {
        // TODO: Validar el token

        var data = requestData.Data;
        var expectedFields = new HashSet<string> { ConnectorUtilsService.FIELD0 };

        var errors = ConnectorUtilsService.CheckErrors(data, expectedFields);

        return errors ?? ResolveGenericOperation(data, token);
    }

    /// <summary>
    /// Resuelve la operación genérica una vez se ha comprobado que tiene los datos necesarios. Busca al tercero en la hoja de Google Sheets y genera la respuesta
    /// con la información necesaria o el error en el caso de que no la encuentre.
    /// </summary>
    /// <param name="data">Datos recibidos en la petición</param>
    /// <param name="token">Token JWT de autenticación</param>
    /// <returns>Los datos del tercero buscado o un error en el caso de que no lo encuentre</returns>
    private IActionResult ResolveGenericOperation(Dictionary<string, FieldModel> data, string token)
    {
        var thirdNif = data[ConnectorUtilsService.FIELD0].Value;

        var thirdData = googleSheetsService.ResolveThirdSearch(thirdNif);

        if (thirdData.Result == null)
        {
            return ConnectorUtilsService.BuildErrorResponse(THIRD_MESSAGE_NOT_FOUND, ConnectorUtilsService.CAUSE_NOT_FOUND, null);
        }

        var responseMap = BuildResponseMap(thirdData.Result!);
        var dataModel = new DataModel(responseMap);

        return connectorUtilsService.GenerateGenericOperationResponse(token, dataModel);
    }

    /// <summary>
    /// Construye la respuesta a la petición de la operación genérica a partir de la lista de datos del tercero
    /// </summary>
    /// <param name="thirdData">Lista de datos del tercero</param>
    /// <returns>El diccionario creado con los datos a enviar en la respuesta de la petición</returns>
    private static Dictionary<string, FieldModel> BuildResponseMap(List<object?> thirdData)
    {
        var map = new Dictionary<string, FieldModel>
        {
            { ConnectorUtilsService.FIELD0, new FieldModel(thirdData[1]?.ToString()) },
            { ConnectorUtilsService.FIELD1, new FieldModel(thirdData[2]?.ToString()) },
            { ConnectorUtilsService.FIELD2, new FieldModel(thirdData[4]?.ToString()) },
            { ConnectorUtilsService.FIELD3, new FieldModel(thirdData[5]?.ToString()) },
            { ConnectorUtilsService.FIELD4, new FieldModel(thirdData[6]?.ToString()) },
            { ConnectorUtilsService.FIELD5, new FieldModel(thirdData[7]?.ToString()) },
            { ConnectorUtilsService.FIELD6, new FieldModel(thirdData[8]?.ToString()) },
            { ConnectorUtilsService.FIELD7, new FieldModel(thirdData[9]?.ToString()) },
            { ConnectorUtilsService.FIELD8, new FieldModel(thirdData[10]?.ToString()) },
            { ConnectorUtilsService.FIELD9, new FieldModel(thirdData[11]?.ToString()) },
        };

        return map;
    }
}