using Microsoft.AspNetCore.Mvc;
using mockup_gestiona_for_developers_net.Connectors.Comm.Service;
using mockup_gestiona_for_developers_net.Connectors.Common.Models;
using mockup_gestiona_for_developers_net.Exception;
using mockup_gestiona_for_developers_net.Models;
using mockup_gestiona_for_developers_net.Service;

namespace mockup_gestiona_for_developers_net.Connectors.IncidenceData.Controller;

[ApiController]
[Route(INCIDENCE_DATA_CONNECTOR_CONTROLLER_PREFIX)]
public class IncidenceDataConnectorController(
    ConnectorUtilsService connectorUtilsService,
    IncidenceService incidenceService) : ControllerBase
{

    private const string INCIDENCE_DATA_CONNECTOR_CONTROLLER_PREFIX = "connectors/incidence-data";

    /// <summary>
    /// Obtiene la versión actual de la API.
    /// </summary>
    /// <param name="token">El token de autenticación</param>
    /// <returns>La versión actual de la API.</returns>
    [HttpGet("versions/current")]
    public IActionResult GetCurrentVersion([FromHeader(Name = "X-Rest-Basic-Token")] string token)
    {
        // TODO: Validar el token

        return connectorUtilsService.GenerateVersionResponse(token);
    }

    /// <summary>
    /// Devuelve la lista de bookmarks
    /// </summary>
    /// <param name="token">El token JWT de autenticación</param>
    /// <returns>El listado de bookmarks del conector</returns>
    [HttpGet("")]
    public IActionResult GetBookmarks([FromHeader(Name = "X-Rest-Basic-Token")] string token)
    {
        // TODO: Validar el token

        var genericOperationUrl = connectorUtilsService.GetGenericOperationUrl(INCIDENCE_DATA_CONNECTOR_CONTROLLER_PREFIX);

        return connectorUtilsService.GenerateBookmarksResponse(token, genericOperationUrl);
    }

    /// <summary>
    /// Recurso operación genérica del conector de datos de incidencia. Devuelve la información relativa a una incidencia que busca a través de su número de expediente
    /// </summary>
    /// <param name="token">Token JWT de autenticación</param>
    /// <param name="requestData">Datos de la incidencia a buscar. `FIELD_0`: número de expediente</param>
    /// <returns>Los datos de la incidencia buscada o un error en el caso de que no la encuentre o exista algún error con los datos recibidos</returns>
    [HttpPost("genericoperations")]
    public async Task<IActionResult> GenericOperation([FromHeader(Name = "X-Rest-Basic-Token")] string token, [FromBody] DataModel requestData)
    {
        // TODO: Validar el token

        var data = requestData.Data;
        var expectedFields = new HashSet<string> { ConnectorUtilsService.FIELD0 };

        var errors = ConnectorUtilsService.CheckErrors(data, expectedFields);

        return errors ?? await ResolveGenericOperation(data, token);
    }

    /// <summary>
    /// Resuelve la operación genérica una vez se ha comprobado que tiene los datos necesarios. Busca la incidencia en la base de datos y genera la respuesta con la información necesaria o el error en el caso de que no la encuentre.
    /// </summary>
    /// <param name="data">Datos recibidos en la petición</param>
    /// <param name="token">Token JWT de autenticación</param>
    /// <returns>Los datos de la incidencia buscada o un error en el caso de que no la encuentre</returns>
    private async Task<IActionResult> ResolveGenericOperation(Dictionary<string, FieldModel> data, string token)
    {
        var fileNumber = data[ConnectorUtilsService.FIELD0].Value;

        try
        {
            var incidenceModel = await incidenceService.SearchIncidence(fileNumber);

            var responseMap = BuildResponseMap(incidenceModel);
            var dataModel = new DataModel(responseMap);

            return connectorUtilsService.GenerateGenericOperationResponse(token, dataModel);
        }
        catch (IncidenceNotFoundException)
        {
            return ConnectorUtilsService.BuildErrorResponse(ConnectorUtilsService.INCIDENCE_MESSAGE_NOT_FOUND, ConnectorUtilsService.CAUSE_NOT_FOUND, null);
        }
    }

    /// <summary>
    /// Construye la respuesta a la petición de la operación genérica a partir del modelo de la incidencia obtenido
    /// </summary>
    /// <param name="incidenceModel">Modelo de la incidencia buscada</param>
    /// <returns>El diccionario creado con los datos a enviar en la respuesta de la petición</returns>
    private static Dictionary<string, FieldModel> BuildResponseMap(IncidenceModel incidenceModel)
    {
        var map = new Dictionary<string, FieldModel>();

        if (incidenceModel.IncidenceNumber != null) map.Add(ConnectorUtilsService.FIELD0, new FieldModel(incidenceModel.IncidenceNumber));
        if (incidenceModel.GestionaRegistryNumber != null) map.Add(ConnectorUtilsService.FIELD1, new FieldModel(incidenceModel.GestionaRegistryNumber));
        if (incidenceModel.Dni != null) map.Add(ConnectorUtilsService.FIELD2, new FieldModel(incidenceModel.Dni));
        if (incidenceModel.Subject != null) map.Add(ConnectorUtilsService.FIELD3, new FieldModel(incidenceModel.Subject));

        return map;
    }
}