using System.Security.Cryptography;
using System.Text;

namespace mockup_gestiona_for_developers_net.Connectors.Comm.Service;

using System.Globalization;
using Common.Exception;
using Configuration;

public class SignatureService(MockupSettings mockupSettings, JwtService jwtService)
{
    private readonly IDictionary<string, string>? _clients = mockupSettings.ConnectorsSettings.Clients;

    private readonly JwtService _jwtService = jwtService ?? throw new ArgumentNullException(nameof(jwtService));

    /// <summary>
    /// Obtiene la firma del UUID a partir del token JWT.
    /// </summary>
    /// <param name="token">Token JWT.</param>
    /// <returns>Firma generada.</returns>
    /// <exception cref="SignatureErrorException">Lanzada si no se puede calcular la firma.</exception>
    public string ObtainSignature(string token)
    {
        try
        {
            var decodedToken = this._jwtService.DecodeJwtToken(token);

            if (!decodedToken.TryGetValue("client_token", out var clientToken) || clientToken is not string clientId)
                throw new InvalidOperationException("El token no contiene el campo 'client_token' o es inválido.");

            if (!decodedToken.TryGetValue("uuid", out var uuidObj) || uuidObj is not string uuid)
                throw new InvalidOperationException("El token no contiene el campo 'uuid' o es inválido.");


            return BuildSignature(clientId, uuid);
        }
        catch (System.Exception ex)
        {
            throw new SignatureErrorException($"Error al obtener la firma: {ex.Message}", ex);
        }
    }

    /// <summary>
    /// Construye la firma a partir del ClientId y el UUID.
    /// </summary>
    /// <param name="clientId">ID del cliente.</param>
    /// <param name="uuid">UUID a firmar.</param>
    /// <returns>Firma generada.</returns>
    /// <exception cref="SignatureErrorException">Lanzada si no se puede calcular la firma.</exception>
    private string BuildSignature(string clientId, string uuid)
    {
        if (!this._clients!.TryGetValue(clientId, out var sharedKey) || string.IsNullOrEmpty(sharedKey))
        {
            throw new SignatureErrorException($"No se encontró la clave compartida para el cliente: {clientId}");
        }

        try
        {
            var hash = Mac256(uuid, sharedKey);
            var hexaHash = ToHexadecimal(hash);
            var signature = EncodeBase64String(hexaHash);

            return signature;
        }
        catch (System.Exception ex)
        {
            throw new SignatureErrorException($"No se ha podido calcular la firma: {ex.Message}", ex);
        }
    }

    /// <summary>
    /// Calcula el HMAC SHA256.
    /// </summary>
    private static byte[] Mac256(string stringToEncode, string sharedKey)
    {
        using var hmac = new HMACSHA256(Encoding.UTF8.GetBytes(sharedKey));
        return hmac.ComputeHash(Encoding.UTF8.GetBytes(stringToEncode));
    }

    /// <summary>
    /// Convierte un array de bytes en un string hexadecimal.
    /// </summary>
    private static string ToHexadecimal(byte[] data)
    {
        var builder = new StringBuilder();
        foreach (var b in data)
        {
            builder.Append(b.ToString("x2", CultureInfo.InvariantCulture));
        }

        return builder.ToString();
    }

    /// <summary>
    /// Codifica un string en Base64.
    /// </summary>
    private static string EncodeBase64String(string data)
    {
        return Convert.ToBase64String(Encoding.UTF8.GetBytes(data));
    }
}