using System.Text.Json.Serialization;
using Microsoft.AspNetCore.Mvc;
using mockup_gestiona_for_developers_net.Connectors.Common.Models;

namespace mockup_gestiona_for_developers_net.Connectors.Comm.Service;

using System.Globalization;

public class ConnectorUtilsService(SignatureService signatureService, IHttpContextAccessor httpContextAccessor)
{
    private const string HEADER_SIGNATURE = "Signature";
    private const string MEDIA_TYPE_GENERIC_OPERATION_RESPONSE = "application/vnd.generic-operation-response+json";
    private const string MEDIA_TYPE_GENERIC_OPERATION_ERROR = "application/vnd.generic-operation-error+json";
    private const string MEDIA_TYPE_BOOKMARK_RESPONSE = "application/vnd.bookmark-list+json";
    private const string MEDIA_TYPE_VERSION_RESPONSE = "application/vnd.version+json";
    private const string MESSAGE_FIELD_ERRORS = "Validation errors in fields.";
    private const string CAUSE_FIELD_ERRORS = "Invalid or missing fields.";
    public const string INCIDENCE_MESSAGE_NOT_FOUND = "Incidence not found";

    private const string CONNECTOR_VERSION = "1.0";

    public const string CAUSE_NOT_FOUND = "ELEMENT_NOT_EXISTS";
    public const string FIELD0 = "FIELD_0";
    public const string FIELD1 = "FIELD_1";
    public const string FIELD2 = "FIELD_2";
    public const string FIELD3 = "FIELD_3";
    public const string FIELD4 = "FIELD_4";
    public const string FIELD5 = "FIELD_5";
    public const string FIELD6 = "FIELD_6";
    public const string FIELD7 = "FIELD_7";
    public const string FIELD8 = "FIELD_8";
    public const string FIELD9 = "FIELD_9";

    /// <summary>
    /// Genera la respuesta HTTP al recurso versión
    /// </summary>
    /// <param name="token">Token JWT</param>
    /// <returns>Respuesta HTTP</returns>
    public IActionResult GenerateVersionResponse(string token)
    {
        var signature = signatureService.ObtainSignature(token);
        var responseContent = new { version = CONNECTOR_VERSION };

        httpContextAccessor.HttpContext?.Response.Headers.Append(HEADER_SIGNATURE, signature);

        var result = new JsonResult(responseContent) { ContentType = MEDIA_TYPE_VERSION_RESPONSE, StatusCode = StatusCodes.Status200OK };

        return result;
    }

    /// <summary>
    /// Genera la respuesta HTTP al recurso bookmarks
    /// </summary>
    /// <param name="token">Token JWT</param>
    /// <param name="genericOperationUrl">Url a la operación genérica</param>
    /// <returns>Respuesta HTTP</returns>
    public IActionResult GenerateBookmarksResponse(string token, string genericOperationUrl)
    {
        var signature = signatureService.ObtainSignature(token);
        var responseContent = new GenericOperationResponse() { GenericOperation = genericOperationUrl };

        httpContextAccessor.HttpContext?.Response.Headers.Append(HEADER_SIGNATURE, signature);

        var result = new JsonResult(responseContent) { ContentType = MEDIA_TYPE_BOOKMARK_RESPONSE, StatusCode = StatusCodes.Status200OK };

        return result;
    }

    /// <summary>
    /// Genera la respuesta HTTP al recurso operación genérica
    /// </summary>
    /// <param name="token">Token JWT de autenticación</param>
    /// <param name="dataModel">Modelo de los datos a enviar en la respuesta</param>
    /// <returns>Respuesta HTTP</returns>
    public IActionResult GenerateGenericOperationResponse(string token, DataModel dataModel)
    {
        var signature = signatureService.ObtainSignature(token);

        httpContextAccessor.HttpContext?.Response.Headers.Append(HEADER_SIGNATURE, signature);

        var result = new JsonResult(dataModel) { ContentType = MEDIA_TYPE_GENERIC_OPERATION_RESPONSE, StatusCode = StatusCodes.Status200OK };

        return result;
    }

    /// <summary>
    /// Construye la respuesta HTTP si existe un error en la petición a la operación genérica
    /// </summary>
    /// <param name="message">Mensaje de error</param>
    /// <param name="cause">Causa del error</param>
    /// <param name="errors">Diccionario con el conjunto de errores que se han encontrado</param>
    /// <returns>Respuesta HTTP con el error</returns>
    public static IActionResult BuildErrorResponse(string message, string cause, Dictionary<string, string>? errors)
    {
        var errorModel = new ErrorModel(message, cause, errors);

        var result = new JsonResult(errorModel) { StatusCode = StatusCodes.Status412PreconditionFailed, ContentType = MEDIA_TYPE_GENERIC_OPERATION_ERROR };

        return result;
    }

    /// <summary>
    /// Comprueba si existen errores en los datos recibidos al recurso operación genérica
    /// </summary>
    /// <param name="data">Datos recibidos en la petición HTTP</param>
    /// <param name="expectedFields">Campos esperados</param>
    /// <param name="checkPriority">Si es verdadero comprueba el formato y valor del campo prioridad, en caso contrario realiza la comprobación normal</param>
    /// <returns>Respuesta HTTP con el mapa de errores en el caso de que existan, o null en caso contrario</returns>
    public static IActionResult? CheckErrors(Dictionary<string, FieldModel> data, HashSet<string> expectedFields, bool checkPriority = false)
    {
        // Validar los parámetros recibidos usando el servicio de validación
        var errors = DataValidatorService.CheckDataParameters(expectedFields, data);

        if (checkPriority)
            CheckPriorityValue(data, errors);

        // Si hay errores, construir y devolver una respuesta con el modelo de error
        return errors.Count > 0 ? BuildErrorResponse(MESSAGE_FIELD_ERRORS, CAUSE_FIELD_ERRORS, errors) : null;
    }

    /// <summary>
    /// Genera la URL al recurso de operación genérica
    /// </summary>
    /// <returns>La URL al recurso operación genérica</returns>
    /// <exception cref="Exception">En el caso de que no pueda generar la URL</exception>
    public string GetGenericOperationUrl(string prefix)
    {
        var request = httpContextAccessor.HttpContext?.Request;
        if (request == null)
        {
            throw new InvalidOperationException("No se pudo obtener la información de la solicitud.");
        }

        var scheme = request.Scheme;
        var host = request.Host;

        return $"{scheme}://{host}/{prefix}/genericoperations";
    }

    /// <summary>
    /// Comprueba el formato y el valor del campo prioridad. Este debe ser un entero comprendido entre 1 y 10 inclusive
    /// </summary>
    /// <param name="data">Datos recibidos en la petición</param>
    /// <param name="errors">Errores obtenidos previamente a esta comprobación</param>
    private static void CheckPriorityValue(Dictionary<string, FieldModel> data, Dictionary<string, string> errors)
    {
        if (errors.ContainsKey(FIELD1)) return;

        try
        {
            var priority = int.Parse(data[FIELD1].Value, CultureInfo.InvariantCulture);
            if (priority is < 1 or > 10)
            {
                errors.Add(FIELD1, DataValidatorService.MSG_FIELD_UNEXPECTED_FORMAT);
            }
        }
        catch (FormatException)
        {
            errors.Add(FIELD1, DataValidatorService.MSG_FIELD_UNEXPECTED_FORMAT);
        }
    }

    internal sealed class GenericOperationResponse
    {
        [JsonPropertyName("generic-operation")] // Define el nombre correcto en el JSON
        public required string GenericOperation { get; set; }
    }
}