package tech.espublico.pades.server.signers.service.sign;

import static tech.espublico.pades.server.signers.service.PDFPrepareHelper.CONTENTS_SIZE;
import static tech.espublico.pades.server.signers.service.PDFPrepareHelper.DIGEST_ALG;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.util.Objects;

import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.DERInteger;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DERObjectIdentifier;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERSet;
import org.bouncycastle.asn1.DERTaggedObject;
import org.bouncycastle.asn1.util.ASN1Dump;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.itextpdf.text.pdf.PdfDictionary;
import com.itextpdf.text.pdf.PdfName;
import com.itextpdf.text.pdf.PdfString;

import tech.espublico.pades.server.helper.IOHelper;
import tech.espublico.pades.server.models.PDFSignedDocumentModel;
import tech.espublico.pades.server.signers.service.digest.hash.PDFHash;
import tech.espublico.pades.server.signers.sign.helper.SignatureHelper;
import tech.espublico.pades.server.signers.sign.timestamp.TimestamperException;
import tech.espublico.pades.server.signers.sign.timestamp.TimestamperHelper;
import tech.espublico.pades.server.signers.sign.timestamp.TimestamperService;
import tech.espublico.pades.server.di.Service;
import tech.espublico.pades.server.di.ServiceLocator;
import tech.espublico.pades.server.exceptions.OpenSignatureException;
import tech.espublico.pades.server.exceptions.OpenSignatureException.OpenSignatureExceptionReason;
import tech.espublico.pades.server.signers.service.digest.DigestUtils;
import tech.espublico.pades.server.signers.sign.exceptions.InvalidSignatureException;

@Service
public class PDFSignService {

	private static final Logger log = LoggerFactory.getLogger(PDFSignService.class);

	public static PDFSignService instance() {
		return ServiceLocator.INSTANCE.getInstance(PDFSignService.class);
	}

	private final TimestamperService timestamperService;

	public PDFSignService() {
		this.timestamperService = TimestamperService.instance();
	}

	/**
	 * Signs the document with the provided encoded pkcs7 data already externally signed
	 *
	 * @param encodedPkcs7
	 * 		The signed data provided previously by this same object with the getBytesToSign and getSecondHashToSign methods
	 *
	 * @return The signed pdf document file and the signer certificate
	 *
	 * @throws OpenSignatureException
	 * 		In case any exception is thrown internally
	 */
	public PDFSignedDocumentModel sign(PDFSignModel pdfSignModel, byte[] encodedPkcs7) throws OpenSignatureException {
		Objects.requireNonNull(encodedPkcs7, "Encoded pkcs7");

		PDFHash pdfHash = pdfSignModel.getPDFHash();

		byte out[] = new byte[CONTENTS_SIZE];
		X509Certificate signerCertificate = null;

		if (encodedPkcs7.length > CONTENTS_SIZE)
			throw new OpenSignatureException("The provided pkcs7 is bigger than the expected content size", OpenSignatureExceptionReason.PKC7_INVALID);

		try {
			signerCertificate = SignatureHelper.validatePKCS7Signature(pdfSignModel.getHash(), encodedPkcs7, pdfSignModel.getByteMode());

		} catch (InvalidSignatureException e) {
			switch (e.getReason()) {
				case CERTIFICATE_EXPIRED:
					throw new OpenSignatureException(e, OpenSignatureExceptionReason.CERTIFICATE_INVALID);

				case CERTIFICATE_NOT_VALID_YET:
					throw new OpenSignatureException(e, OpenSignatureExceptionReason.CERTIFICATE_INVALID);

				case COULD_NOT_PARSE_PKCS7:
					throw new OpenSignatureException(e, OpenSignatureExceptionReason.CERTIFICATE_IO_ERROR);

				case EMPTY_CERTIFICATE_CHAIN:
					throw new OpenSignatureException(e, OpenSignatureExceptionReason.CERTIFICATE_IO_ERROR);

				case INVALID_KEY:
					throw new OpenSignatureException(e, OpenSignatureExceptionReason.SIGN_INVALID_KEY);

				case INVALID_SIGNATURE:
					throw new OpenSignatureException(e, OpenSignatureExceptionReason.SIGN_INVALID);

				case SIGNER_NOT_FOUND:
					throw new OpenSignatureException(e, OpenSignatureExceptionReason.SIGN_INVALID);

				case UNKNOWN:
				default:
					throw new OpenSignatureException(e, OpenSignatureExceptionReason.UNKNOWN);
			}
		}

		// Copies the encoded pkcs7 signed data to a temp byte array
		System.arraycopy(encodedPkcs7, 0, out, 0, encodedPkcs7.length);

		if (log.isDebugEnabled()) {
			log.debug("CONTENTS_SIZE=" + CONTENTS_SIZE);
			log.debug("encodedPkcs7.length=" + encodedPkcs7.length);
		}

		try {

			// Creates the signed dictionary with the provided encoded pkcs7
			// data
			PdfDictionary dic2 = new PdfDictionary();
			dic2.put(PdfName.CONTENTS, new PdfString(out).setHexWriting(true));

			// Closes the document with the signature, this actually writes the
			// signature in the pdf document file
			pdfHash.getPdfSignatureAppearance().close(dic2);
		} catch (Exception e) {
			throw new OpenSignatureException("Could not close/write the pdf document when signing", e, OpenSignatureExceptionReason.PDF_IO_ERROR);
		}

		return new PDFSignedDocumentModel(pdfHash.getSignedPdf().toFile(), signerCertificate);
	}

	/**
	 * Returns the encoded pkcs7 data with the provided signed bytes
	 *
	 * @param signatureBytes
	 * 		signature byte array
	 * @param signedAttributes
	 * 		signed attributes
	 *
	 * @return an encoded pkcs7 byte array with the provided information
	 *
	 * @throws OpenSignatureException
	 * 		In case of any error
	 */
	public byte[] getRncodedPkcs7(PDFSignModel pdfSignModel, byte[] signatureBytes, ASN1EncodableVector signedAttributes, final Boolean requestTimeStamp)
			throws OpenSignatureException {

		signedAttributes.get(0).getDERObject();


		byte[] encodedPkcs7 = null;

		// Create the set of Hash algorithms

		// digest algos
		ASN1EncodableVector algos = new ASN1EncodableVector();
		algos.add(new DERObjectIdentifier(DIGEST_ALG.getOid()));
		algos.add(new DERNull());
		DERSet digestAlgorithms = new DERSet(new DERSequence(algos));
		// digestAlgorithms.addObject(new DERSequence(algos));

		// Create the contentInfo.
		ASN1EncodableVector ev = new ASN1EncodableVector();
		ev.add(new DERObjectIdentifier("1.2.840.113549.1.7.1")); // PKCS7SignedData

		DERSequence contentinfo = new DERSequence(ev);

		// Get all the certificates
		//
		ASN1EncodableVector v = new ASN1EncodableVector();

		// Create signerinfo structure.
		//
		ASN1EncodableVector signerinfo = new ASN1EncodableVector();

		DERSet dercertificates = null;
		try {
			Certificate[] certs = pdfSignModel.getCerts();
			ByteArrayInputStream bais = null;
			ASN1InputStream tempstream = null;
			for (int c = 0; c < pdfSignModel.getCerts().length; c++) {
				try {
					bais = new ByteArrayInputStream(certs[c].getEncoded());
					tempstream = new ASN1InputStream(bais);
					v.add(tempstream.readObject());
				} finally {
					IOHelper.closeQuietly(tempstream);
					IOHelper.closeQuietly(bais);
				}
			}

			dercertificates = new DERSet(v);

			// Add the signerInfo version
			//
			signerinfo.add(new DERInteger(1));

			v = new ASN1EncodableVector();
			v.add(getIssuer((X509Certificate) certs[0]));

			v.add(new DERInteger(((X509Certificate) certs[0]).getSerialNumber()));
			signerinfo.add(new DERSequence(v));

			// Add the digestAlgorithm
			v = new ASN1EncodableVector();
			v.add(new DERObjectIdentifier(DIGEST_ALG.getOid()));
			v.add(new DERNull());
			signerinfo.add(new DERSequence(v));

			// add the authenticated attribute if present
			signerinfo.add(new DERTaggedObject(false, 0, new DERSet(signedAttributes)));

			// Add the digestEncryptionAlgorithm
			v = new ASN1EncodableVector();
			v.add(new DERObjectIdentifier("1.2.840.113549.1.1.1"));// RSA
			v.add(new DERNull());
			signerinfo.add(new DERSequence(v));

			// Add the encrypted digest
			signerinfo.add(new DEROctetString(signatureBytes));

		} catch (CertificateEncodingException e) {
			throw new OpenSignatureException(e, OpenSignatureExceptionReason.CERTIFICATE_INVALID_ENCODING);
		} catch (IOException e) {
			throw new OpenSignatureException(e, OpenSignatureExceptionReason.CERTIFICATE_IO_ERROR);
		}

		// Add unsigned attributes (timestamp)
		if (Boolean.TRUE.equals(requestTimeStamp) //
				&& pdfSignModel.getTimestamperData() != null //
				&& pdfSignModel.getTimestamperData().isEnabled()) {//
			try {
				byte[] timestampHash = DigestUtils.digest(DIGEST_ALG, signatureBytes);
				ASN1EncodableVector unsignedAttributes = //
						buildUnsignedAttributes(pdfSignModel, DIGEST_ALG.getName(), timestampHash);
				if (unsignedAttributes != null) {
					signerinfo.add(new DERTaggedObject(false, 1, new DERSet(unsignedAttributes)));
				}
			} catch (TimestamperException e) {
				throw new OpenSignatureException(e, OpenSignatureExceptionReason.TIMESTAMP_ERROR);
			} catch (IOException e) {
				throw new OpenSignatureException(e, OpenSignatureExceptionReason.TIMESTAMP_IO_ERROR); // Could not read token as ASN1 object
			}
		}

		// Finally build the body out of all the components above
		ASN1EncodableVector body = new ASN1EncodableVector();
		body.add(new DERInteger(1)); // pkcs7 version, always 1
		body.add(digestAlgorithms);
		body.add(contentinfo);
		body.add(new DERTaggedObject(false, 0, dercertificates));
		// Only allow one signerInfo
		body.add(new DERSet(new DERSequence(signerinfo)));

		// Now we have the body, wrap it in it's PKCS7Signed shell
		// and return it
		//
		ASN1EncodableVector whole = new ASN1EncodableVector();
		whole.add(new DERObjectIdentifier("1.2.840.113549.1.7.2"));// PKCS7_SIGNED_DATA
		whole.add(new DERTaggedObject(0, new DERSequence(body)));

		if (log.isDebugEnabled())
			log.debug(ASN1Dump.dumpAsString(new DERSequence(whole)));

		try {
			encodedPkcs7 = new DERSequence(whole).getEncoded();
		} catch (IOException e) {
			throw new OpenSignatureException(e, OpenSignatureExceptionReason.DER_SEQUENCE_IO_ERROR);
		}// <---
		return encodedPkcs7;

	}

	private ASN1EncodableVector buildUnsignedAttributes(PDFSignModel pdfSignModel, String digestAlg, byte[] hash) throws TimestamperException, IOException {
		byte[] respBytes = TimestamperHelper.getRespBytesFromTimestamper(//
						this.timestamperService, //
						digestAlg, hash, pdfSignModel.getTimestamperData(),//
						pdfSignModel.getBackupTimestamperData());//

		if (respBytes == null) {
			return null;
		}

		ByteArrayInputStream bais = null;
		ASN1InputStream tempstream = null;
		try {
			// Parse timestamp token as an ASN1 object
			bais = new ByteArrayInputStream(respBytes);
			tempstream = new ASN1InputStream(bais);

			// id-aa-timeStampToken OBJECT IDENTIFIER ::= { iso(1) member-body(2)
			// us(840) rsadsi(113549) pkcs(1) pkcs-9(9) smime(16) aa(2) 14 }
			// SignatureTimeStampToken ::= TimeStampToken

			// time Stamp token : id-aa-timeStampToken da RFC3161, alias old
			// id-smime-aa-timeStampToken

			ASN1Sequence seq = (ASN1Sequence) tempstream.readObject();
			DERObject timeStampToken = (DERObject) seq.getObjectAt(1);

			ASN1EncodableVector v = new ASN1EncodableVector();

			v.add(new DERObjectIdentifier("1.2.840.113549.1.9.16.2.14")); // id-aa-timeStampToken
			v.add(new DERSet(timeStampToken));

			ASN1EncodableVector unsignedAttributes = new ASN1EncodableVector();
			unsignedAttributes.add(new DERSequence(v));

			return unsignedAttributes;
		} finally {
			IOHelper.closeQuietly(tempstream);
			IOHelper.closeQuietly(bais);
		}
	}

	private static DERObject getIssuer(X509Certificate cert) throws CertificateEncodingException, IOException {

		byte[] abTBSCertificate = cert.getTBSCertificate();
		ASN1Sequence seq = (ASN1Sequence) readDERObject(abTBSCertificate);
		return (DERObject) seq.getObjectAt(seq.getObjectAt(0) instanceof DERTaggedObject ? 3 : 2);

	}

	static DERObject readDERObject(byte[] ab) throws IOException {

		ByteArrayInputStream bais = null;
		BufferedInputStream bis = null;
		ASN1InputStream in = null;
		try {
			bais = new ByteArrayInputStream(ab);
			bis = new BufferedInputStream(bais);
			in = new ASN1InputStream(bis);

			DERObject obj = in.readObject();

			return obj;

		} finally {
			IOHelper.closeQuietly(in);
			IOHelper.closeQuietly(bis);
			IOHelper.closeQuietly(bais);
		}
	}
}
