package tech.espublico.pades.server.signers.service.prepare;

import java.io.BufferedOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.ClosedChannelException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.pdf.BaseFont;
import com.itextpdf.text.pdf.ICC_Profile;
import com.itextpdf.text.pdf.PdfArray;
import com.itextpdf.text.pdf.PdfDictionary;
import com.itextpdf.text.pdf.PdfICCBased;
import com.itextpdf.text.pdf.PdfName;
import com.itextpdf.text.pdf.PdfReader;
import com.itextpdf.text.pdf.PdfStamper;
import com.itextpdf.text.pdf.PdfStream;
import com.itextpdf.text.pdf.PdfString;
import com.itextpdf.text.pdf.PdfWriter;

import tech.espublico.pades.server.helper.FilesHelper;
import tech.espublico.pades.server.helper.IOHelper;
import tech.espublico.pades.server.signers.service.PDFPrepareHelper;
import tech.espublico.pades.server.signers.service.PDFSignatureModel;
import tech.espublico.pades.server.di.Service;
import tech.espublico.pades.server.di.ServiceLocator;
import tech.espublico.pades.server.exceptions.OpenSignatureException;
import tech.espublico.pades.server.exceptions.OpenSignatureException.OpenSignatureExceptionReason;
import tech.espublico.pades.server.services.style.ColorProfileService;
import tech.espublico.pades.server.services.style.FontService;

@Service
public class PDFPrepareForSignatureService {
	private static final Logger log = LoggerFactory.getLogger(PDFPrepareForSignatureService.class);

	public static PDFPrepareForSignatureService instance() {
		return ServiceLocator.INSTANCE.getInstance(PDFPrepareForSignatureService.class);
	}

	private static final int SIGNATURES_PER_PAGE_HORIZONTAL = 3;
	private static final int SIGNATURES_PER_PAGE_VERTICAL = 3;

	public Path prepare(PDFSignatureModel pdfSignatureModel) throws OpenSignatureException {

		final Path unsignedPath = pdfSignatureModel.getUnsignedPath();
		final int totalSignatures = pdfSignatureModel.getTotalSignatures();

		if (Files.notExists(unsignedPath))
			throw new RuntimeException(String.format("The File in is required and must be a valid and existing file [%s]", unsignedPath.toString()));

		if (totalSignatures == 0)
			throw new RuntimeException("Total Signatures must be more than zero");

		final String downloadText = pdfSignatureModel.getDownloadText();

		final Path preparedPdf;
		try {
			preparedPdf = FilesHelper.createTempFile("preparedForSigning", ".pdf").toPath();
		} catch (IOException e) {
			throw new OpenSignatureException(e, OpenSignatureExceptionReason.DOCUMENT_CANT_CREATE_TEMP_FILE);
		}

		PdfReader pdfReader = null;

		OutputStream outputStream = null;
		BufferedOutputStream bufferedOutputStream = null;
		PdfStamper pdfStamper = null;

		try {

			// Get the reader
			pdfReader = PDFPrepareHelper.createPDFReader(unsignedPath);

			outputStream = Files.newOutputStream(preparedPdf, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.CREATE);//
			bufferedOutputStream = new BufferedOutputStream(outputStream);
			pdfStamper = new PdfStamper(pdfReader, bufferedOutputStream);

			final Rectangle pageRect = pdfReader.getPageSizeWithRotation(1);
			float pageWidth = pageRect.getWidth();
			float pageHeight = pageRect.getHeight();

			boolean isHorizontal = PDFPrepareHelper.isHorizontal(pdfReader);
			BaseFont tahomaFont = PDFPrepareHelper.loadFont(FontService.instance().getFont1());
			BaseFont arialFont = PDFPrepareHelper.loadFont(FontService.instance().getFont2());

			PdfWriter pdfWriter = pdfStamper.getWriter();
			pdfWriter.setPDFXConformance(PdfWriter.PDFA1A);
			pdfWriter.setCompressionLevel(PdfStream.BEST_COMPRESSION);

			// Embeds encoding and color profile to make PDF/A-1 compliant
			{
				PdfDictionary catalog = pdfReader.getCatalog();
				PdfDictionary acroForm = (PdfDictionary) PdfReader.getPdfObject(catalog.get(PdfName.ACROFORM), catalog);
				if (acroForm == null) {
					acroForm = new PdfDictionary();
					catalog.put(PdfName.ACROFORM, acroForm);
				}
				PdfDictionary dr = (PdfDictionary) PdfReader.getPdfObject(acroForm.get(PdfName.DR), acroForm);
				if (dr == null) { // Ensure DR exists in the acroform
					dr = new PdfDictionary();
					acroForm.put(PdfName.DR, dr);
				}

				PdfDictionary encodings = new PdfDictionary();
				dr.put(PdfName.ENCODING, encodings);
				PdfDictionary dic = new PdfDictionary(PdfName.PDFDOCENCODING);
				dic.put(PdfName.DIFFERENCES, new PdfArray());

				PdfDictionary outi = new PdfDictionary(PdfName.OUTPUTINTENT);
				outi.put(PdfName.OUTPUTCONDITIONIDENTIFIER, new PdfString("sRGB IEC61966-2.1"));
				outi.put(PdfName.INFO, new PdfString("sRGB IEC61966-2.1"));
				outi.put(PdfName.S, PdfName.GTS_PDFA1);

				try (InputStream is = FilesHelper.newInputStream(ColorProfileService.instance().getColorProfile())) {
					ICC_Profile icc = ICC_Profile.getInstance(is);
					PdfICCBased ib = new PdfICCBased(icc);
					ib.remove(PdfName.ALTERNATE);
					outi.put(PdfName.DESTOUTPUTPROFILE, pdfWriter.addToBody(ib).getIndirectReference());
				}
				pdfWriter.getExtraCatalog().put(PdfName.OUTPUTINTENTS, new PdfArray(outi));
			}

			{ // Add needed pages for signatures

				int lastPage = pdfReader.getNumberOfPages() + 1;
				int SIGNATURES_PER_PAGE = isHorizontal ? SIGNATURES_PER_PAGE_HORIZONTAL : SIGNATURES_PER_PAGE_VERTICAL;
				int pagesToAdd = (int) Math.ceil((float) totalSignatures / (float) SIGNATURES_PER_PAGE) - pdfReader.getNumberOfPages();

				for (int index = 0; index < pagesToAdd; index++) {
					pdfStamper.insertPage(lastPage, pageRect);
				}
			}

			PDFPrepareBarcode//
					.addBarcodeAndDownloadTextInAllPages(pdfSignatureModel, pdfStamper, pdfReader, pageHeight, pageWidth, arialFont);

			PDFPrepareMark//
					.addMarkLabels(pdfSignatureModel, pdfStamper, pdfWriter, pdfReader, pageHeight, pageWidth, tahomaFont);

			// Creates signature fields
			PDFPrepareSignatureFields//
					.addSignatureFields(pdfSignatureModel, //
							pdfStamper, pdfWriter, pageHeight, pageWidth, isHorizontal, pdfReader.getPageRotation(1) % 360);
		} catch (FileNotFoundException e) {
			throw new OpenSignatureException(e, OpenSignatureExceptionReason.DOCUMENT_OUTPUT_FILE_NOT_FOUND);
		} catch (IOException e) {
			throw new OpenSignatureException(e, OpenSignatureExceptionReason.DOCUMENT_IO_ERROR);
		} catch (DocumentException e) {
			throw new OpenSignatureException(e, OpenSignatureExceptionReason.DOCUMENT_PDF_SETUP_ERROR);
		} finally {
			closeQuietly(pdfStamper);
			IOHelper.closeQuietly(outputStream);
			IOHelper.closeQuietly(bufferedOutputStream);
			if (pdfReader != null)
				pdfReader.close();
		}

		return preparedPdf;
	}

	private static void closeQuietly(PdfStamper closeable) {
		if (closeable == null)
			return;
		try {
			closeable.close();
		} catch (ClosedChannelException ignore) {

		} catch (IOException e) {
			log.warn("close quietly (no too much, review it please) {} IOException: {}", closeable, e);
		} catch (DocumentException e) {
			log.warn("close quietly (no too much, review it please) {} DocumentException: {}", closeable, e);
		}
	}
}
