package tech.espublico.pades.server.signers.service.prepare;

import java.awt.Point;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Image;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.pdf.BaseFont;
import com.itextpdf.text.pdf.PdfContentByte;
import com.itextpdf.text.pdf.PdfReader;
import com.itextpdf.text.pdf.PdfStamper;
import com.itextpdf.text.pdf.PdfTemplate;

import tech.espublico.pades.server.helper.StringUtils;
import tech.espublico.pades.server.signers.service.PDFPrepareHelper;
import tech.espublico.pades.server.signers.service.PDFSignatureModel;
import tech.espublico.pades.server.signers.service.PDFSignatureModel.BarcodeType;

public class PDFPrepareBarcode {
	static void addBarcodeAndDownloadTextInAllPages(//
			PDFSignatureModel pdfSignatureModel, //
			PdfStamper pdfStamper, //
			PdfReader pdfReader, //
			float height, float width, //
			BaseFont font) //
			throws IOException, DocumentException {

		final String modelDownloadText = pdfSignatureModel.getDownloadText();
		if (modelDownloadText == null)
			return;

		final Path barcodeImage = pdfSignatureModel.getBarcodeImage();
		if (barcodeImage == null || !Files.exists(barcodeImage))
			return;

		final Image barcode = Image.getInstance(barcodeImage.toFile().getCanonicalPath());
		final BarcodeType barcodeType = pdfSignatureModel.getBarcodeType();

		final String pageCountLabel = pdfSignatureModel.getPageCountLabel();

		final int numberOfPages = pdfReader.getNumberOfPages();
		final int pageRotation = pdfReader.getPageRotation(1) % 360;
		final int rotation = 90 + pageRotation;

		for (int index = 1; index <= numberOfPages; index++) {

			/*
			 *  Bar code
			 */

			Image image = Image.getInstance(barcode);

			float barcode_x = getBarcodeX(barcodeType, width, image);
			float barcode_y = getBarcodeY(barcodeType);

			image.setAbsolutePosition(barcode_x, barcode_y);
			image.setRotationDegrees(rotation);

			pdfStamper.getOverContent(index).addImage(image);

			/*
			 *  CSV text
			 */

			float verificationText_x = getVerificationTextX(barcodeType, width);
			float verificationText_y = getVerificationTextY(barcodeType, image);

			Rectangle rectangle = PDFPrepareHelper//
					.getRectangle((int) verificationText_x, (int) verificationText_y, width, height, pageRotation);

			int rectHtmlWidth = (int) (rectangle.getRight() - rectangle.getLeft());
			int rectHtmlHeight = (int) (rectangle.getTop() - rectangle.getBottom());

			//Template will be transformed to image (to be able to rotate text)
			PdfTemplate template = pdfStamper.getOverContent(index).createTemplate(rectHtmlWidth, rectHtmlHeight);
			template.rectangle(rectangle);

			// Text to put the template
			String downloadText = modelDownloadText;
			if (StringUtils.isNotEmpty(pageCountLabel)) {
				downloadText += String.format(pageCountLabel, index, numberOfPages);
			}
			String[] texts = downloadText.split("\n");

			// Draw text in the template
			int fontSize = PDFPrepareHelper.VERIFICATION_TEXT_FONT_SIZE;
			template.setFontAndSize(font, fontSize);

			template.beginText();

			template.moveText(0, 0);
			int line = 1;
			for (String text : texts) {
				int increment = line * fontSize;
				Point position = PDFPrepareHelper.getPosition(rectHtmlWidth, rectHtmlHeight, pageRotation, increment);
				template.showTextAlignedKerned(PdfContentByte.ALIGN_LEFT, text, position.x, position.y, rotation);
				line++;
				template.newlineText();
			}
			template.endText();

			// Transform the template to image
			Image templateImage = Image.getInstance(template);
			templateImage.setAbsolutePosition(verificationText_x, verificationText_y);

			pdfStamper.getOverContent(index).addImage(templateImage);
		}
	}

	private static float getBarcodeX(BarcodeType type, float documentWidth, Image image) {
		switch (type) {
			case QR:
				return documentWidth - image.getHeight() - PDFPrepareHelper.MARGIN_RIGHT_PTS + PDFPrepareHelper.QR_DEFAULT_MARGIN;
			case BARCODE:
				return documentWidth - PDFPrepareHelper.BARCODE_MARGIN;
			default:
				throw new IllegalArgumentException("Unhandled barcode type: " + type);
		}
	}

	private static float getBarcodeY(BarcodeType type) {
		switch (type) {
			case QR:
				return PDFPrepareHelper.MARGIN_BOTTOM - PDFPrepareHelper.QR_DEFAULT_MARGIN;
			case BARCODE:
				return PDFPrepareHelper.MARGIN_BOTTOM;
			default:
				throw new IllegalArgumentException("Unhandled barcode type: " + type);
		}
	}

	private static float getVerificationTextX(BarcodeType type, float documentWidth) {
		switch (type) {
			case QR:
			case BARCODE:
				return documentWidth - PDFPrepareHelper.VERIFICATION_TEXT_MARGIN_RIGHT;
			default:
				throw new IllegalArgumentException("Unhandled barcode type: " + type);
		}
	}

	private static float getVerificationTextY(BarcodeType type, Image barcodeImage) {
		switch (type) {
			case QR:
				return barcodeImage.getWidth() + PDFPrepareHelper.MARGIN_BOTTOM + 4;
			case BARCODE:
				return PDFPrepareHelper.MARGIN_BOTTOM;
			default:
				throw new IllegalArgumentException("Unhandled barcode type: " + type);
		}
	}
}
