package tech.espublico.pades.server.signers.service.finalize;

import java.util.Map.Entry;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.function.Function;

import org.cliffc.high_scale_lib.NonBlockingHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * V and K must honor equals and hashcode
 */
public class TimedHashMap<K, V> {

	private static final Logger log = LoggerFactory.getLogger(TimedHashMap.class);

	/**
	 * Relative reset expiry time after get call. Absolute does not reset expiry time after get call
	 */
	public enum TimeOutMode {
		Relative,
		Absolute}

	private final ConcurrentHashMap<K, TimedValue<V>> map;
	private final long timeout;
	private final TimeOutMode timeoutMode;
	private final String name;
	private final Timer timer;

	/**
	 * Creates a map that removes the object from itself once the time passes. Also executes a finalize action.
	 *
	 * @param name
	 * 		name of the map
	 * @param expiryMillis
	 * 		expiration in miliseconds of the objects in the map
	 * @param timeoutMode
	 * 		expiration mode relarive (relative to the last access of the object) absoulte (from the time the object was inserted into the map)
	 * @param finalize
	 * 		action to run once the expiration passes, before removing the object from the map.
	 */
	public TimedHashMap(String name, long expiryMillis, TimeOutMode timeoutMode, Consumer<V> finalize) {
		this.map = new ConcurrentHashMap<>();
		this.name = name;
		this.timeout = expiryMillis;
		this.timeoutMode = timeoutMode;
		this.timer = new Timer("TimedHashMap." + getName(), true);
		TimerTask timerTask = new TimerTask() {
			public void run() {
				for (Entry<K, TimedValue<V>> e : map.entrySet()) {
					TimedValue<V> value = e.getValue();
					if (!value.alive()) {
						try {
							//Si falla la finalización pasamos al siguiente elemento
							finalize.accept(value.value);
							map.remove(e.getKey(), value);
						} catch (Throwable ignore) {
							log.warn("Failed to finalize {}", e.getKey(), ignore);
						}
					}
				}
			}
		};
		this.timer.scheduleAtFixedRate(timerTask, expiryMillis / 2, expiryMillis / 2);
	}

	/**
	 * Creates a Map without finalize actions
	 *
	 * @param name
	 * 		name of the map
	 * @param expiryMillis
	 * 		expiration of the objects in the map
	 * @param timeoutMode
	 * 		timeout mode relative or absoulte
	 */
	public TimedHashMap(String name, long expiryMillis, TimeOutMode timeoutMode) {
		this(name, expiryMillis, timeoutMode, v -> {
		});
	}

	public V put(K key, V value) {
		if (null == value)
			throw new IllegalArgumentException("null value");
		map.put(key, new TimedValue<V>(value, timeout));
		return value;
	}

	public V get(K key) {
		for (int i = 0; i < 2; i++) {
			TimedValue<V> entry = map.get(key);
			if (entry == null)
				return null;
			if (TimeOutMode.Relative.equals(timeoutMode)) {
				if (map.replace(key, entry, entry.renew(timeout)))
					return entry.value;
				continue;
			}
			if (entry.alive())
				return entry.value;
			if (map.remove(key, entry))
				return null;
		}
		return null;
	}

	public boolean containsKey(K key) {
		return map.get(key) != null;
	}

	public V remove(K key) {
		TimedValue<V> entry = map.remove(key);
		if (entry == null)
			return null;
		return entry.value;
	}

	public V computeIfAbsent(K sessionId, Function<K, V> create) {
		return this.map.computeIfAbsent(sessionId, k -> {
			return new TimedValue<>(create.apply(k), this.timeout);
		}).value;
	}

	public void clear() {
		map.clear();
	}

	public void stop() {
		if (timer != null)
			timer.cancel();
	}

	public String getName() {
		return name;
	}

	private final static class TimedValue<V> {
		private final long deadLine;
		private final V value;

		TimedValue(V value, long timeout) {
			this.value = value;
			deadLine = System.currentTimeMillis() + timeout;
		}

		boolean alive() {
			return deadLine > System.currentTimeMillis();
		}

		TimedValue<V> renew(long timout) {
			return new TimedValue<V>(value, timout);
		}

		public int hashCode() {
			return (int) (deadLine ^ (deadLine >>> 32));
		}

		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			return deadLine == ((TimedValue<?>) obj).deadLine;
		}
	}

	public int size() {
		return map.size();
	}
}
