package tech.espublico.pades.server.signers.service.digest.second;

import static org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers.id_aa_signingCertificate;
import static org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers.id_aa_signingCertificateV2;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.util.Calendar;

import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.DERInteger;
import org.bouncycastle.asn1.DERObjectIdentifier;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERSet;
import org.bouncycastle.asn1.DERTaggedObject;
import org.bouncycastle.asn1.DERUTCTime;
import org.bouncycastle.asn1.cms.Attribute;
import org.bouncycastle.asn1.cms.IssuerAndSerialNumber;
import org.bouncycastle.asn1.ess.ESSCertID;
import org.bouncycastle.asn1.ess.ESSCertIDv2;
import org.bouncycastle.asn1.ess.SigningCertificate;
import org.bouncycastle.asn1.ess.SigningCertificateV2;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.IssuerSerial;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.ocsp.OCSPResp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import sun.security.util.DerOutputStream;
import sun.security.util.DerValue;
import sun.security.util.ObjectIdentifier;
import tech.espublico.pades.server.helper.IOHelper;
import tech.espublico.pades.server.models.CertificateOCSPConfiguration;
import tech.espublico.pades.server.models.PDFSignerModel.ByteMode;
import tech.espublico.pades.server.signers.sign.DigestAlgorithm;
import tech.espublico.pades.server.signers.sign.helper.CertificateHelper;
import tech.espublico.pades.server.signers.sign.ocsp.OCSPHttpClientException;
import tech.espublico.pades.server.signers.sign.ocsp.OCSPService;
import tech.espublico.pades.server.signers.sign.ocsp.OCSPValidationException;
import tech.espublico.pades.server.di.Service;
import tech.espublico.pades.server.di.ServiceLocator;
import tech.espublico.pades.server.exceptions.OpenSignatureException;
import tech.espublico.pades.server.exceptions.OpenSignatureException.OpenSignatureExceptionReason;
import tech.espublico.pades.server.signers.service.PDFPrepareHelper;
import tech.espublico.pades.server.signers.service.digest.DigestUtils;

@Service
public class PDFSecondHashService {

	private static final Logger log = LoggerFactory.getLogger(PDFSecondHashService.class);

	public static PDFSecondHashService instance() {
		return ServiceLocator.INSTANCE.getInstance(PDFSecondHashService.class);
	}

	private final OCSPService ocspService;

	public PDFSecondHashService() {
		this.ocspService = OCSPService.instance();
	}

	public PDFSecondHash getSecondHashToSign(PDFSecondHashModel pdfSecondHashModel, byte[] hash) throws OpenSignatureException {

		ASN1EncodableVector signedAttributes = //
				buildSignedAttributes(pdfSecondHashModel, hash);

		byte[] bytesForSecondHash = null;

		try {
			bytesForSecondHash = new DERSet(signedAttributes).getEncoded();
		} catch (IOException e) {
			throw new OpenSignatureException("Could not obtain second hash to sign", e, OpenSignatureExceptionReason.HASH_INVALID_DER_SET);
		}

		PDFSecondHash pdfSecondHash;
		switch (pdfSecondHashModel.getByteMode()) {
			case RAW:
				pdfSecondHash = new PDFSecondHash(signedAttributes, bytesForSecondHash);
				break;
			case HASH: {
				byte[] secondHash = DigestUtils.digest(PDFPrepareHelper.DIGEST_ALG, bytesForSecondHash);

				byte[] bytesToSign = null;
				try {
					bytesToSign = //
							getBytesToSign(new ObjectIdentifier(PDFPrepareHelper.DIGEST_ALG.getOid()), secondHash);
				} catch (IOException e) {
					throw new OpenSignatureException("Could not obtain second hash to sign", e, OpenSignatureExceptionReason.HASH_GET_BYTES_TO_SIGN);
				}

				pdfSecondHash = new PDFSecondHash(signedAttributes, bytesToSign);
				break;
			}
			default:
				throw new IllegalArgumentException("Unexpected byteModel property");
		}
		return pdfSecondHash;
	}

	private ASN1EncodableVector buildSignedAttributes(//
			PDFSecondHashModel pdfSecondHashModel, //
			byte[] hash) //
			throws OpenSignatureException {

		Calendar signingTime = pdfSecondHashModel.getSigningDate();

		ASN1EncodableVector signedAttributes = new ASN1EncodableVector();

		// Content type
		ASN1EncodableVector v = new ASN1EncodableVector();
		v.add(new DERObjectIdentifier("1.2.840.113549.1.9.3"));// CONTENT_TYPE
		v.add(new DERSet(new DERObjectIdentifier("1.2.840.113549.1.7.1")));// PKCS7_DATA
		signedAttributes.add(new DERSequence(v));
		if (signingTime == null) {
			log.debug("No signing time!");
		} else {
			log.debug("[buildSignedAttributes]:: cal: " + signingTime);
			// signing time
			v = new ASN1EncodableVector();
			v.add(new DERObjectIdentifier("1.2.840.113549.1.9.5")); // SIGNING_TIME
			v.add(new DERSet(new DERUTCTime(signingTime.getTime())));
			signedAttributes.add(new DERSequence(v));
		}

		// message digest
		v = new ASN1EncodableVector();
		v.add(new DERObjectIdentifier("1.2.840.113549.1.9.4"));// MESSAGE_DIGEST
		v.add(new DERSet(new DEROctetString(hash)));
		signedAttributes.add(new DERSequence(v));

		X509Certificate signingCert = getSigningCertificate(pdfSecondHashModel);
		byte[] certHash = getCertHash(PDFPrepareHelper.DIGEST_ALG, signingCert);

		// add signing-certificate
		if (pdfSecondHashModel.getCerts().length == 1) {
			if (PDFPrepareHelper.DIGEST_ALG.equals(DigestAlgorithm.SHA1)) {
				SigningCertificate sc = new SigningCertificate(new ESSCertID(certHash));
				signedAttributes.add(new Attribute(id_aa_signingCertificate, new DERSet(sc)));
			} else {
				ESSCertIDv2 essCert = new ESSCertIDv2(new AlgorithmIdentifier(PDFPrepareHelper.DIGEST_ALG.getOid()), certHash);
				SigningCertificateV2 scv2 = new SigningCertificateV2(new ESSCertIDv2[] { essCert });
				signedAttributes.add(new Attribute(id_aa_signingCertificateV2, new DERSet(scv2)));
			}
		} else if (pdfSecondHashModel.getCerts().length > 1) {
			// add signing-certificate with issuer-serial
			// X509Certificate issuer = (X509Certificate) model.getCerts()[1];
			X509CertificateHolder cholder;
			try {
				cholder = new X509CertificateHolder(signingCert.getEncoded());
			} catch (Exception e) {
				throw new OpenSignatureException("Could not get encoded issuer", e, OpenSignatureExceptionReason.SIGN_CANT_CREATE_SIGNATURE);
			}
			IssuerAndSerialNumber issuerAndSerialNumber = cholder.getIssuerAndSerialNumber();
			GeneralNames gn = new GeneralNames(new GeneralName(issuerAndSerialNumber.getName()));
			DERInteger serial = issuerAndSerialNumber.getSerialNumber();
			IssuerSerial issuerSerial = new IssuerSerial(gn, serial);
			if (PDFPrepareHelper.DIGEST_ALG.equals(DigestAlgorithm.SHA1)) {
				ESSCertID essCertID = new ESSCertID(certHash, issuerSerial);
				SigningCertificate signingCertificate = new SigningCertificate(essCertID);
				signedAttributes.add(new Attribute(id_aa_signingCertificate, new DERSet(signingCertificate)));
			} else {
				ESSCertIDv2 essCertIdv2 = new ESSCertIDv2(new AlgorithmIdentifier(PDFPrepareHelper.DIGEST_ALG.getOid()), certHash, issuerSerial);
				SigningCertificateV2 signingCertificateV2 = new SigningCertificateV2(new ESSCertIDv2[] { essCertIdv2 });
				signedAttributes.add(new Attribute(id_aa_signingCertificateV2, new DERSet(signingCertificateV2)));
			}
		}

		// Add signed attribute for OCSP Responses
		{
			DERSequence ocspInfo = getRevocationInfoArchival(//
					pdfSecondHashModel,//
					pdfSecondHashModel.getCerts()[0], //
					pdfSecondHashModel.getCerts(), //
					signingTime);//

			if (ocspInfo != null) { // Only add if there has been an ocsp check
				signedAttributes.add(ocspInfo);
			}
		}

		return signedAttributes;

	}

	private static byte[] getBytesToSign(ObjectIdentifier algOid, byte[] digest) throws IOException {

		DerOutputStream out = null;
		DerOutputStream algS = null;
		DerOutputStream oidS = null;
		DerOutputStream nullS = null;
		DerOutputStream digestS = null;
		try {
			out = new DerOutputStream();

			algS = new DerOutputStream();

			oidS = new DerOutputStream();
			oidS.putOID(algOid);

			nullS = new DerOutputStream();
			nullS.putNull();

			algS.putSequence(new DerValue[] { new DerValue(oidS.toByteArray()), new DerValue(nullS.toByteArray()) });

			digestS = new DerOutputStream();
			digestS.putOctetString(digest);

			DerValue[] dv = new DerValue[] { new DerValue(algS.toByteArray()), new DerValue(digestS.toByteArray()) };
			out.putSequence(dv);

			return out.toByteArray();
		} finally {
			IOHelper.closeQuietly(oidS);
			IOHelper.closeQuietly(nullS);
			IOHelper.closeQuietly(algS);
			IOHelper.closeQuietly(digestS);
			IOHelper.closeQuietly(out);
		}

	}

	private DERSequence getRevocationInfoArchival(//
			PDFSecondHashModel pdfSecondHashModel, //
			Certificate certificate, //
			Certificate[] certChain, //
			Calendar signingTime) //
			throws OpenSignatureException {

		if (pdfSecondHashModel.getSignerOCSPConfiguration() == null) {
			log.debug("Skipping OCSP check because there is no configuration");
			return null; // No signer OCSP url
		}

		CertificateOCSPConfiguration ocspConfiguration = //
				pdfSecondHashModel.getSignerOCSPConfiguration();
		if (log.isDebugEnabled())
			log.debug("Checking OCSP with configuration {}", ocspConfiguration);

		if (!ocspConfiguration.isCheckOCSP()) {
			log.debug("Skipping ocsp check");
			return null;
		}
		X509Certificate x509Certificate = (X509Certificate) certificate;

		X509Certificate issuerCertificate = CertificateHelper.getIssuerCertificate(x509Certificate, certChain);
		if (issuerCertificate == null) {
			throw new OpenSignatureException("OCSP UNKNOWN ERROR: could not determine issuer certificate", OpenSignatureExceptionReason.OCSP_UNKNOWN);
		}

		OCSPResp resp;
		try {
			resp = this.ocspService.validateOCSPResponse(//
					x509Certificate, //
					signingTime.getTime(), //
					pdfSecondHashModel.getSignerOCSPConfiguration(), //
					issuerCertificate);//
		} catch (OCSPHttpClientException e) {
			throw new OpenSignatureException("OCSP CLIENT ERROR", e, OpenSignatureExceptionReason.OCSP_CLIENT_ERROR,
					pdfSecondHashModel.getSignerOCSPConfiguration().getServerURL());
		} catch (OCSPValidationException e) {
			throw new OpenSignatureException("OCSP VALIDATION ERROR", e, OpenSignatureExceptionReason.OCSP_INVALID,
					pdfSecondHashModel.getSignerOCSPConfiguration().getServerURL());
		} catch (Throwable e) {
			throw new OpenSignatureException("OCSP UNKNOWN ERROR", e, OpenSignatureExceptionReason.OCSP_UNKNOWN,
					pdfSecondHashModel.getSignerOCSPConfiguration().getServerURL());
		}

		if (resp == null)
			return null;

		log.debug("RESP STATUS=" + resp.getStatus());

		OCSPResp[] ocspList = new OCSPResp[] { resp };

		ASN1EncodableVector v = new ASN1EncodableVector();
		v.add(new DERObjectIdentifier("1.2.840.113583.1.1.8"));// RevocationInfoArchival
		ASN1EncodableVector list = new ASN1EncodableVector();
		for (OCSPResp or : ocspList) {
			ByteArrayInputStream bais = null;
			ASN1InputStream t = null;
			try {
				bais = new ByteArrayInputStream(or.getEncoded());
				t = new ASN1InputStream(bais);
				list.add(t.readObject());
			} catch (IOException e) {
				e.printStackTrace();
			} finally {
				IOHelper.closeQuietly(t);
				IOHelper.closeQuietly(bais);
			}
		}
		v.add(new DERSet(new DERSequence(new DERTaggedObject(true, 1, new DERSequence(list)))));
		return new DERSequence(v);
	}

	private static X509Certificate getSigningCertificate(PDFSecondHashModel pdfSecondHashModel) throws OpenSignatureException {
		// !
		return (X509Certificate) pdfSecondHashModel.getCerts()[0];
	}

	private static byte[] getCertHash(DigestAlgorithm alg, X509Certificate cert) throws OpenSignatureException {
		try {
			return DigestUtils.digest(alg, cert.getEncoded());
		} catch (CertificateEncodingException e) {
			throw new OpenSignatureException("Could not get encoded cert", e, OpenSignatureExceptionReason.SIGN_CANT_CREATE_SIGNATURE);
		}
	}
}
