package tech.espublico.pades.server.signers.service.digest.hash;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateEncodingException;

import org.apache.commons.codec.binary.Hex;
import org.bouncycastle.util.encoders.HexEncoder;

import com.itextpdf.text.Chunk;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Element;
import com.itextpdf.text.Font;
import com.itextpdf.text.Image;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.pdf.BaseFont;
import com.itextpdf.text.pdf.ColumnText;
import com.itextpdf.text.pdf.PdfReader;
import com.itextpdf.text.pdf.PdfSignatureAppearance;
import com.itextpdf.text.pdf.PdfTemplate;

import tech.espublico.pades.server.helper.FilesHelper;
import tech.espublico.pades.server.exceptions.OpenSignatureException;
import tech.espublico.pades.server.exceptions.OpenSignatureException.OpenSignatureExceptionReason;
import tech.espublico.pades.server.services.style.FontService;
import tech.espublico.pades.server.signers.service.PDFPrepareHelper;

public class PDFSignatureAppearance {

	public static void createSignatureAppearance(//
			PDFDigestModel pdfDigestModel, PdfReader reader, PdfSignatureAppearance pdfSignatureAppearance) //
			throws OpenSignatureException {
		pdfSignatureAppearance.setAcro6Layers(true);
		pdfSignatureAppearance.setVisibleSignature("Signature" + pdfDigestModel.getSignatureNumber());
		pdfSignatureAppearance.setImageScale(-1); // Do not scale image

		if (pdfDigestModel.isCertifyNoChangesAllowed()) {
			pdfSignatureAppearance.setCertificationLevel(PdfSignatureAppearance.CERTIFIED_NO_CHANGES_ALLOWED);
		}

		{
			BaseFont bfont = null;
			try {
				bfont = BaseFont.createFont(FontService.instance().getFont5(), BaseFont.CP1252, BaseFont.EMBEDDED);
			} catch (Exception e) {
				throw new OpenSignatureException(e, OpenSignatureExceptionReason.SIGN_FONT_NOT_FOUND);
			}

			Font font = new Font(bfont);
			font.setSize(PDFPrepareHelper.SIGNATURE_FIELD_FONT_SIZE);

			String signerNameAndCount = pdfDigestModel.getUserName();
			if (pdfDigestModel.getSignatureCountLabel() != null)
				signerNameAndCount += " " + String.format(//
						pdfDigestModel.getSignatureCountLabel(), //
						pdfDigestModel.getSignatureNumber(), pdfDigestModel.getTotalSignatures());//

			StringBuilder text = new StringBuilder();
			text.append("\n" + pdfDigestModel.getUserCharge());
			text.append("\n" + pdfDigestModel.getSignatureDate());
			try {
				MessageDigest m = MessageDigest.getInstance("MD5");
				m.reset();
				m.update(pdfDigestModel.getCerts()[0].getEncoded());
				text.append("\nHASH: ");
				text.append(Hex.encodeHexString(m.digest()));
			} catch (NoSuchAlgorithmException e1) {
				e1.printStackTrace(); // !!,
			} catch (CertificateEncodingException e) {
				throw new OpenSignatureException(e, OpenSignatureExceptionReason.CERTIFICATE_INVALID_ENCODING);
			}

			Image image = null;
			if (FilesHelper.exists(pdfDigestModel.getSignatureImage())) {
				try {
					image = Image.getInstance(pdfDigestModel.getSignatureImage().toFile().getCanonicalPath());
				} catch (Exception e) {
					throw new OpenSignatureException(e, OpenSignatureExceptionReason.SIGN_INVALID_IMAGE_ELEMENT);
				}
			}

			createSignatureAppearance(//
					pdfDigestModel,//
					font, //
					signerNameAndCount, text.toString(), image, reader.getPageRotation(1) % 360, pdfSignatureAppearance);
		}
	}
	private static void createSignatureAppearance(
			PDFDigestModel pdfDigestModel,
			Font font, String name, String text, Image image, int rotation, PdfSignatureAppearance sap)
			throws OpenSignatureException {

		try {
			// Layer 0
			PdfTemplate layer0 = sap.getLayer(0);
			layer0.setBoundingBox(new Rectangle(100, 100));
			layer0.setLiteral("% DSBlank\n");

			// Layer 2
			PdfTemplate layer2 = sap.getLayer(2);
			Rectangle sapRect = sap.getRect();
			Rectangle rect = new Rectangle(sapRect.getHeight(), sapRect.getHeight());
			layer2.setBoundingBox(rect);

			int correction = 0;
			if (rotation > 0)
				correction = (int) (sapRect.getHeight() - PDFPrepareHelper.SIGNATURE_FIELD_WIDTH);

			int SIGNATURE_FIELD_IMAGE_MAX_HEIGHT = (int) (sapRect.getHeight() * PDFPrepareHelper.SIGNATURE_FIELD_IMAGE_MAX_HEIGHT_PERCENT);
			float SIGNATURE_FIELD_IMAGE_MAX_WIDTH = sapRect.getWidth() - correction - (PDFPrepareHelper.SIGNATURE_FIELD_MARGIN * 2);

			// origin is the bottom-left
			Rectangle imageRect = new Rectangle(//
					PDFPrepareHelper.SIGNATURE_FIELD_MARGIN, //
					PDFPrepareHelper.SIGNATURE_FIELD_MARGIN, //
					SIGNATURE_FIELD_IMAGE_MAX_HEIGHT, //
					rect.getWidth() - PDFPrepareHelper.SIGNATURE_FIELD_MARGIN);
			Rectangle textRect = new Rectangle(//
					SIGNATURE_FIELD_IMAGE_MAX_HEIGHT + (PDFPrepareHelper.SIGNATURE_FIELD_MARGIN * 2), //
					0, //
					rect.getHeight() - PDFPrepareHelper.SIGNATURE_FIELD_MARGIN, //
					rect.getHeight() - PDFPrepareHelper.SIGNATURE_FIELD_MARGIN);

			int runDirection = sap.getRunDirection();

			ColumnText ct2 = new ColumnText(layer2);
			ct2.setRunDirection(runDirection);
			ct2.setSimpleColumn(imageRect.getLeft(), imageRect.getBottom(), imageRect.getRight(), imageRect.getTop(), 0, Element.ALIGN_CENTER);
			if (pdfDigestModel.isVisibleSignature() && image != null) {

				Image im = Image.getInstance(image);
				im.scaleToFit(SIGNATURE_FIELD_IMAGE_MAX_HEIGHT, SIGNATURE_FIELD_IMAGE_MAX_WIDTH);

				float x = 0, y = 0;
				if (im.getScaledHeight() == SIGNATURE_FIELD_IMAGE_MAX_WIDTH) {
					x = (SIGNATURE_FIELD_IMAGE_MAX_HEIGHT - im.getScaledWidth()) / 2;
				} else {
					y = -(SIGNATURE_FIELD_IMAGE_MAX_WIDTH - im.getScaledHeight() - PDFPrepareHelper.SIGNATURE_FIELD_MARGIN) / 2;
				}
				y -= correction + PDFPrepareHelper.SIGNATURE_FIELD_MARGIN * 4;
				ct2.addElement(new Chunk(im, x, y, false));
			}
			ct2.go();

			if (pdfDigestModel.isVisibleSignature()) {
				ColumnText ct = new ColumnText(layer2);
				ct.setRunDirection(runDirection);
				font.setStyle(Font.BOLD);
				ct.setSimpleColumn(new Phrase(name, font), textRect.getLeft(), textRect.getBottom(), textRect.getRight(), textRect.getTop() - correction,
						font.getSize(), Element.ALIGN_LEFT);
				ct.go();

				ct = new ColumnText(layer2);
				ct.setRunDirection(runDirection);
				font.setStyle(Font.NORMAL);
				ct.setSimpleColumn(new Phrase(text, font), textRect.getLeft(), textRect.getBottom(), textRect.getRight(), textRect.getTop() - correction,
						font.getSize(), Element.ALIGN_LEFT);
				ct.go();
			}

			// Top Layer
			PdfTemplate frm = sap.getTopLayer();
			frm.setBoundingBox(rect);

			if (rotation == 0)
				frm.concatCTM(0, 1, -1, 0, rect.getHeight(), 0);
			else if (rotation == 90)
				frm.concatCTM(-1, 0, 0, -1, rect.getWidth(), rect.getHeight());
			else if (rotation == 180)
				frm.concatCTM(0, -1, 1, 0, 0, rect.getWidth());

			frm.addTemplate(layer0, 0, 0);
			frm.addTemplate(layer2, 0, 0);

		} catch (DocumentException e) {
			throw new OpenSignatureException(e, OpenSignatureExceptionReason.APPEARANCE_ERROR);
		}
	}
}
