package tech.espublico.pades.server.signers.service.digest.hash;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.cert.X509Certificate;
import java.util.HashMap;

import com.itextpdf.text.pdf.PdfDate;
import com.itextpdf.text.pdf.PdfDictionary;
import com.itextpdf.text.pdf.PdfName;
import com.itextpdf.text.pdf.PdfNumber;
import com.itextpdf.text.pdf.PdfPKCS7;
import com.itextpdf.text.pdf.PdfReader;
import com.itextpdf.text.pdf.PdfSignatureAppearance;
import com.itextpdf.text.pdf.PdfStamper;
import com.itextpdf.text.pdf.PdfString;

import tech.espublico.pades.server.helper.ByteArrayHelper;
import tech.espublico.pades.server.helper.CollectionsHelper;
import tech.espublico.pades.server.helper.FilesHelper;
import tech.espublico.pades.server.helper.StringUtils;
import tech.espublico.pades.server.di.Service;
import tech.espublico.pades.server.di.ServiceLocator;
import tech.espublico.pades.server.exceptions.OpenSignatureException;
import tech.espublico.pades.server.exceptions.OpenSignatureException.OpenSignatureExceptionReason;
import tech.espublico.pades.server.signers.service.PDFPrepareHelper;
import tech.espublico.pades.server.signers.service.digest.DigestUtils;

@Service
public class PDFHashService {

	public static PDFHashService instance() {
		return ServiceLocator.INSTANCE.getInstance(PDFHashService.class);
	}

	public PDFHash getHash(//
			PDFDigestModel pdfDigestModel, Path preparedPath)//
			throws OpenSignatureException {

		if (CollectionsHelper.isEmpty(pdfDigestModel.getCerts()))
			throw new RuntimeException("certchain required");

		if (Files.notExists(preparedPath))
			throw new RuntimeException("The File in is required and must be a valid and existing file");

		if (StringUtils.isEmpty(pdfDigestModel.getReason()))
			throw new RuntimeException("A reason must be provided to sign");

		if (StringUtils.isEmpty(pdfDigestModel.getLocation()))
			throw new RuntimeException("A location must be provided to sign");

		if (pdfDigestModel.getSignatureNumber() == 0)
			throw new RuntimeException("Signature number must be more than zero");

		if (pdfDigestModel.getSignDate() == null)
			throw new RuntimeException("Sign date must be provided");

		byte[] bytesToSign = null;

		try {
			PdfReader pdfReader = PDFPrepareHelper.createPDFReader(preparedPath);
			Path signedPdf;
			OutputStream outputStream;
			BufferedOutputStream bufferedOutputStream;
			PdfStamper pdfStamper;
			PdfSignatureAppearance pdfSignatureAppearance;

			byte[] contentHash;

			try {
				signedPdf = FilesHelper.createTempFile("signed", ".pdf").toPath();
				outputStream = Files.newOutputStream(signedPdf);
				bufferedOutputStream = new BufferedOutputStream(outputStream);
			} catch (IOException e) {
				throw new OpenSignatureException(e, OpenSignatureExceptionReason.DOCUMENT_CANT_CREATE_TEMP_FILE);
			}

			try {
				pdfStamper = PdfStamper.createSignature(pdfReader, bufferedOutputStream, '\0', null, true);
			} catch (Exception e) {
				throw new OpenSignatureException(e, OpenSignatureExceptionReason.SIGN_CANT_CREATE_SIGNATURE);
			} // Always appending signatures

			// Get the signature appeareance, throught a pdf stamper
			pdfSignatureAppearance = pdfStamper.getSignatureAppearance();
			pdfSignatureAppearance.setCrypto(null, pdfDigestModel.getCerts(), null, PdfSignatureAppearance.WINCER_SIGNED); // set the external
			// certification
			// flag
			pdfSignatureAppearance.setSignDate(pdfDigestModel.getSignDate()); // Set the sign date
			pdfSignatureAppearance.setReason(pdfDigestModel.getReason()); // Set the signing reason
			pdfSignatureAppearance.setLocation(pdfDigestModel.getLocation()); // Set the signing location

			// Sets the signature image and signer data
			PDFSignatureAppearance.createSignatureAppearance(pdfDigestModel, pdfReader, pdfSignatureAppearance);

			// Sets the digerster information
			pdfSignatureAppearance.setExternalDigest(new byte[128], new byte[20], "RSA");

			{ // Set the crypto dictionary, which contains the signature user
				// information
				PdfDictionary dic = new PdfDictionary();
				dic.put(PdfName.FT, PdfName.SIG);
				dic.put(PdfName.FILTER, new PdfName("Adobe.PPKLite"));
				dic.put(PdfName.R, new PdfNumber(65541));

				dic.put(PdfName.SUBFILTER, new PdfName("adbe.pkcs7.detached"));
				dic.put(PdfName.M, new PdfDate(pdfDigestModel.getSignDate()));

				dic.put(PdfName.NAME, new PdfString(PdfPKCS7.getSubjectFields((X509Certificate) pdfDigestModel.getCerts()[0]).getField("CN")));
				dic.put(PdfName.REASON, new PdfString(pdfDigestModel.getReason()));
				dic.put(PdfName.LOCATION, new PdfString(pdfDigestModel.getLocation()));
				dic.put(PdfName.CONTACTINFO, new PdfString(pdfDigestModel.getContactInfo()));

				pdfSignatureAppearance.setCryptoDictionary(dic);
			}

			// Set the signature content size (this reserves the needed bytes
			// for the signature)
			HashMap<PdfName, Integer> exc = new HashMap<>();
			exc.put(PdfName.CONTENTS, new Integer(PDFPrepareHelper.CONTENTS_SIZE * 2 + 2));

			// Sets the data needed to create the hash to be signed
			try {
				pdfSignatureAppearance.preClose(exc);
			} catch (Exception e) {
				throw new OpenSignatureException(e, OpenSignatureExceptionReason.SIGN_CANT_PRECLOSE_DOCUMENT);
			}

			PDFHash pdfHash = new PDFHash();
			pdfHash.setPdfReader(pdfReader);
			pdfHash.setSignedPdf(signedPdf);
			pdfHash.setOutputStream(outputStream);
			pdfHash.setBufferedOutputStream(bufferedOutputStream);
			pdfHash.setPdfStamper(pdfStamper);
			pdfHash.setPdfSignatureAppearance(pdfSignatureAppearance);
			pdfHash.setContentHash(getContentHash(pdfSignatureAppearance));

			return pdfHash;

		} catch (IllegalArgumentException e) {
			if (e.getMessage().equalsIgnoreCase("PdfReader not opened with owner password")) {
				// http://1t3xt.info/tutorials/faq.php?branch=faq.itext&node=password
				throw new OpenSignatureException("Cannot sign restricted PDF file", e, OpenSignatureExceptionReason.PDF_RESTRICTED);
			}
			throw new OpenSignatureException(e, OpenSignatureExceptionReason.UNKNOWN);
		}
	}

	private static byte[] getContentHash(PdfSignatureAppearance sap) throws OpenSignatureException {
		try (InputStream inputStream = sap.getRangeStream()) {
			return DigestUtils.digest(PDFPrepareHelper.DIGEST_ALG, ByteArrayHelper.streamToByteArray(inputStream));
		} catch (IOException e) {
			throw new OpenSignatureException(e, OpenSignatureExceptionReason.SIGN_CANT_GET_CONTENT_BYTES_TO_SIGN);
		}
	}


}
