package tech.espublico.pades.server.signers.service;

import java.nio.file.Path;
import java.security.Security;

import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import com.itextpdf.text.FontFactory;

import tech.espublico.pades.server.models.PDFSignedDocumentModel;
import tech.espublico.pades.server.signers.service.digest.hash.PDFDigestModel;
import tech.espublico.pades.server.signers.service.digest.hash.PDFHash;
import tech.espublico.pades.server.signers.service.digest.hash.PDFHashService;
import tech.espublico.pades.server.signers.service.digest.second.PDFSecondHash;
import tech.espublico.pades.server.signers.service.digest.second.PDFSecondHashModel;
import tech.espublico.pades.server.signers.service.digest.second.PDFSecondHashService;
import tech.espublico.pades.server.di.Service;
import tech.espublico.pades.server.di.ServiceLocator;
import tech.espublico.pades.server.exceptions.OpenSignatureException;
import tech.espublico.pades.server.signers.service.prepare.PDFPrepareForSignatureService;
import tech.espublico.pades.server.signers.service.sign.PDFSignModel;
import tech.espublico.pades.server.signers.service.sign.PDFSignService;

@Service
public class PDFSignatureService {
	/**
	 * Static initialization, sets the security provider
	 */
	static {
		Security.addProvider(new BouncyCastleProvider());
		FontFactory.defaultEmbedding = true;
	}

	public static PDFSignatureService instance() {
		return ServiceLocator.INSTANCE.getInstance(PDFSignatureService.class);
	}

	private final PDFPrepareForSignatureService pdfPrepareForSignatureService;
	private final PDFHashService pdfHashService;
	private final PDFSecondHashService pdfSecondHashService;
	private final PDFSignService pdfSignService;

	public PDFSignatureService() {
		this.pdfPrepareForSignatureService = PDFPrepareForSignatureService.instance();
		this.pdfHashService = PDFHashService.instance();
		this.pdfSecondHashService = PDFSecondHashService.instance();
		this.pdfSignService = PDFSignService.instance();
	}

	public Path preparePdf(PDFSignatureModel pdfSignatureModel) throws OpenSignatureException {
		return this.pdfPrepareForSignatureService.prepare(pdfSignatureModel);
	}

	public PDFHash calculateHash(PDFDigestModel pdfDigestModel, Path preparePdf) throws OpenSignatureException {
		return this.pdfHashService.getHash(pdfDigestModel, preparePdf);
	}

	public PDFSecondHash calculateSecondHash(PDFSecondHashModel pdfSecondHashModel, byte[] hash) throws OpenSignatureException {
		return this.pdfSecondHashService.getSecondHashToSign(pdfSecondHashModel, hash);
	}

	public byte[] wrapHashPkcs7(PDFSignModel pdfSignModel, byte[] signatureBytes, ASN1EncodableVector signedAttributes, boolean requestTimeStamp)
			throws OpenSignatureException {
		return this.pdfSignService.getRncodedPkcs7(pdfSignModel, signatureBytes, signedAttributes, requestTimeStamp);
	}

	public PDFSignedDocumentModel sign(PDFSignModel pdfSignModel, byte[] encodedPkcs7) throws OpenSignatureException {
		return this.pdfSignService.sign(pdfSignModel, encodedPkcs7);
	}
}
