package tech.espublico.pades.server.services.validation;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.security.SignatureException;
import java.util.ArrayList;
import java.util.List;

import javax.security.auth.x500.X500Principal;

import org.apache.commons.lang3.exception.ExceptionUtils;

import com.itextpdf.text.pdf.AcroFields;
import com.itextpdf.text.pdf.PdfDictionary;
import com.itextpdf.text.pdf.PdfName;
import com.itextpdf.text.pdf.PdfObject;
import com.itextpdf.text.pdf.PdfPKCS7;
import com.itextpdf.text.pdf.PdfReader;

import tech.espublico.pades.server.di.Service;
import tech.espublico.pades.server.di.ServiceLocator;
import tech.espublico.pades.server.services.validation.SignatureState.Reason;

@Service
public class SignatureValidationService {

	public static SignatureValidationService instance() {
		return ServiceLocator.INSTANCE.getInstance(SignatureValidationService.class);
	}

	public ValidationResponse validate(Path path, List<String> expectedSignatures) throws ValidationException {

		ValidationResponse validationResponse = new ValidationResponse();
		PdfReader reader = null;
		try (InputStream inputStream = Files.newInputStream(path, StandardOpenOption.READ)){
			reader = new PdfReader(inputStream);

			PdfObject dssDictionary = reader.getCatalog().get(new PdfName("DSS"));
			validationResponse.setHasDssObject(dssDictionary != null);

			AcroFields acroFields = reader.getAcroFields();
			List<String> signatures = acroFields.getSignatureNames();

			if (signatures.size() != expectedSignatures.size()) {
				throw new ValidationException(String.format("Expected signatures cannot be compare (expexcted %s, found %s)", expectedSignatures.size(), signatures.size()));
			}

			List<SignatureState> signatureStates = new ArrayList<>();
			for (int i = 0; i < signatures.size(); i++) {
				String signature = signatures.get(i);
				String expectedSignature = expectedSignatures.get(i);
				SignatureState signatureState = new SignatureState(signature, expectedSignature);

				PdfDictionary signatureDict = acroFields.getSignatureDictionary(signature);
				PdfPKCS7 pk = acroFields.verifySignature(signature);
				String signatureDate = signatureDict.getAsString(PdfName.M).toString();
				signatureState.setSignatureDate(signatureDate);

				try {
					if (!pk.verify()) {
						signatureState.setValid(false);
						signatureState.setReason(Reason.INVALID);
						signatureStates.add(signatureState);
						continue;
					}
				} catch (SignatureException e) {
					signatureState.setValid(false);
					signatureState.setReason(Reason.VALIDATION_ERROR);
					signatureStates.add(signatureState);
					signatureState.setValidationErrorTrace(ExceptionUtils.getStackTrace(e));
					continue;
				}

				X500Principal principal = pk.getSigningCertificate().getSubjectX500Principal();
				String cn = principal.getName(X500Principal.CANONICAL);

				if (expectedSignature.equals(cn)) {
					if (!signatureDict.contains(PdfName.CONTENTS)) {
						signatureState.setValid(false);
						signatureState.setReason(Reason.EMPTY);
						signatureState.setSignature(cn);
						signatureStates.add(signatureState);
						continue;
					}
				} else {
					signatureState.setValid(false);
					signatureState.setReason(Reason.NOT_EQUALS);
					signatureState.setSignature(cn);
					signatureStates.add(signatureState);
					continue;
				}

				signatureStates.add(signatureState);
			}

			validationResponse.setSignatureStateList(signatureStates);
		} catch (IOException e) {
			throw new ValidationException(String.format("Cannot open %s", path), e);
		} finally {
			if (reader != null)
				reader.close();
		}

		return validationResponse;
	}

}
