package tech.espublico.pades.server.services.timestamper;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.NoSuchAlgorithmException;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.tsp.TSPException;
import org.bouncycastle.tsp.TimeStampRequest;
import org.bouncycastle.tsp.TimeStampRequestGenerator;
import org.bouncycastle.tsp.TimeStampResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tech.espublico.pades.server.helper.DateEx;
import tech.espublico.pades.server.helper.IOHelper;
import tech.espublico.pades.server.signers.sign.timestamp.TimestamperException;
import tech.espublico.pades.server.signers.sign.timestamp.TimestamperException.TimestamperExceptionReason;
import tech.espublico.pades.server.di.Service;

@Service
public class TimestamperService implements tech.espublico.pades.server.signers.sign.timestamp.TimestamperService {

	private static final Logger log = LoggerFactory.getLogger(TimestamperService.class);

	@Override
	public byte[] getTSResponse(String digestAlg, byte[] hash, URL serverTimestamp, String user, String password) throws TimestamperException {

		HttpURLConnection connection = null;

		try {

			byte[] timestampResponseBytes = null;
			TimeStampRequest timestampRequest = createTimestampRequest(digestAlg, hash);

			Map<String, String> headers = new HashMap<>();
			headers.put("Content-Type", "application/timestamp-query");
			headers.put("Content-Transfer-Encoding", "binary");
			if (StringUtils.isNotEmpty(user)) {
				String userPassword = user + ":" + password;
				headers.put("Authorization", "Basic " + Base64.getEncoder().encodeToString(userPassword.getBytes()));
			}

			OutputStream out = null;

			try {
				connection = HttpClientHelper.createConnection(serverTimestamp, headers, "POST");
				out = connection.getOutputStream();
				out.write(timestampRequest.getEncoded());

			} catch (IOException e) {
				throw new TimestamperException(String.format("Could not post timestamp request to server %s", serverTimestamp), serverTimestamp,
						TimestamperExceptionReason.COULD_NOT_CONNECT, e);
			} finally {
				IOHelper.closeQuietly(out);
			}

			InputStream fis = null;
			ByteArrayOutputStream baos = null;
			try {
				int responseCode = connection.getResponseCode();

				if (HttpURLConnection.HTTP_OK == responseCode) {

					try {
						fis = connection.getInputStream();
					} catch (IOException e) {
						throw new TimestamperException(String.format("Could not get input stream from connection with server %s", serverTimestamp),
								serverTimestamp, TimestamperExceptionReason.COULD_NOT_CONNECT, e);
					}

					baos = new ByteArrayOutputStream();
					IOHelper.copy(fis, baos);

					timestampResponseBytes = baos.toByteArray();

					if (StringUtils.equalsIgnoreCase(connection.getContentEncoding(), "base64")) {
						timestampResponseBytes = Base64.getDecoder().decode(new String(timestampResponseBytes));
					}

					try {
						TimeStampResponse timeStampResponse = new TimeStampResponse(timestampResponseBytes);
						if (timeStampResponse.getStatus() != 0)
							throw new TimestamperException("Invalid timestamp response " + timeStampResponse.getStatusString(), serverTimestamp,
									TimestamperExceptionReason.INVALID_RESPONSE);
						timeStampResponse.validate(timestampRequest);
					} catch (TSPException e) {
						throw new TimestamperException(String.format("Invalid timestamp response from server %s", serverTimestamp), serverTimestamp,
								TimestamperExceptionReason.INVALID_RESPONSE, e);
					} catch (IOException e) {
						throw new TimestamperException(String.format("Could not parse timestamp response from server %s", serverTimestamp), serverTimestamp,
								TimestamperExceptionReason.INVALID_RESPONSE, e);
					}

					return timestampResponseBytes;

				} else if (HttpURLConnection.HTTP_BAD_REQUEST == responseCode) {
					throw new TimestamperException(String.format("BAD REQUEST FROM %s", serverTimestamp), serverTimestamp,
							TimestamperExceptionReason.BAD_REQUEST);

				} else {
					throw new TimestamperException(String.format("ERROR CODE %d FROM SERVER %s", responseCode, serverTimestamp), serverTimestamp,
							TimestamperExceptionReason.UNKNOWN_ERROR);
				}

			} catch (IOException e) {
				throw new TimestamperException(String.format("Could not read timestamp response from server %s", serverTimestamp), serverTimestamp,
						TimestamperExceptionReason.COULD_NOT_READ_RESPONSE, e);

			} finally {
				IOHelper.closeQuietly(fis);
				IOHelper.closeQuietly(baos);
			}

		} finally {
			HttpClientHelper.disconnectQuietly(connection);
		}
	}

	protected TimeStampRequest createTimestampRequest(String digestAlg, byte[] hash) {
		TimeStampRequestGenerator tsqGenerator = new TimeStampRequestGenerator();
		tsqGenerator.setCertReq(true);

		BigInteger nonce = BigInteger.valueOf(DateEx.getDateGMT0().getTime());

		TimeStampRequest tsrequest;
		try {
			tsrequest = tsqGenerator.generate(DigestAlgorithm.getByName(digestAlg).getOid(), hash, nonce);
		} catch (NoSuchAlgorithmException e) {
			throw new IllegalArgumentException("cannot find oid for digest alg " + digestAlg);
		}
		return tsrequest;
	}
}
