package tech.espublico.pades.server.services.timestamper;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tech.espublico.pades.server.helper.IOHelper;
import tech.espublico.pades.server.helper.StringUtils;
import tech.espublico.pades.server.helper.WriteReadOutputStream;

public class HttpClientHelper {
	
	private final static Logger log = LoggerFactory.getLogger(HttpClientHelper.class);

	public final static String HTTP_AGENT_KEY = "User-Agent";
	public final static String HTTP_AGENT = "HttpAgent/0.1";
	
	public final static String CONTENT_TYPE_KEY = "Content-Type";
	public final static String CONTENT_TYPE = "application/x-www-form-urlencoded; charset=UTF-8";
	
	public final static String HTTP_ACCEPT_KEY = "ACCEPT";
	public final static String HTTP_ACCEPT = "*/*";
	
	public final static String HTTP_CONTENT_LENGTH_KEY = "Content-Length";
	public final static String HTTP_LOCATION_HEADER = "Location";
	
	private final static int DEFAULT_CONNECTION_TIMEOUT = 10000;
	private final static int DEFAULT_READ_TIMEOUT = 10000;
	
	public static HttpURLConnection createConnection(URL url, Map<String, String> properties, String method) throws IOException {
		
		if (url == null)
			throw new IllegalArgumentException("Url cannot be null");
		
		if (properties == null)
			properties = new HashMap<String, String>();
		
		HttpURLConnection conn = (HttpURLConnection) url.openConnection();

		conn.setDoInput(true);
		conn.setDoOutput(true);
		conn.setUseCaches(false);
		conn.setRequestMethod(method);
		conn.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
		conn.setReadTimeout(DEFAULT_READ_TIMEOUT);
		
		if (!properties.containsKey(HTTP_AGENT_KEY))
			properties.put(HTTP_AGENT_KEY, HTTP_AGENT);
		
		if (!properties.containsKey(HTTP_ACCEPT_KEY))
			properties.put(HTTP_ACCEPT_KEY, HTTP_ACCEPT);
			
		if (!properties.containsKey(CONTENT_TYPE_KEY))
			properties.put(CONTENT_TYPE_KEY, CONTENT_TYPE);
		
		Iterator<String> keyset = properties.keySet().iterator();
		while(keyset.hasNext()) {
			String key = keyset.next();
			conn.addRequestProperty(key, properties.get(key));
		}
		
		return conn;
	}
	
	
	public static HttpURLConnection createConnection(URL url, String method) throws IOException {
		return createConnection(url, null, method);
	}
	
	public static int sendRequest(HttpURLConnection connection, InputStream content) throws IOException, ReadResponseTimeoutException {
		return sendRequest(connection, null,content);
	}
	
	public static int sendRequest(HttpURLConnection connection, String contentType, InputStream content) throws IOException, ReadResponseTimeoutException {
		if (connection == null)
			throw new IllegalArgumentException("Connection cannot be null");

		if (content == null)
			throw new IllegalArgumentException("InputStream cannot be null");
		
		try {
			if ( content.available() > 0 ) {
				Integer lenght = content.available();
				connection.addRequestProperty("Content-Length", lenght.toString());
			}
		} catch (IOException e) {
			throw new IOException("Could not read inputstream size", e);
		}
		
		if ( !StringUtils.isEmpty(contentType) ) {
			connection.addRequestProperty("Content-Type", contentType);
		}
		
	
		OutputStream os = null;
		try {
			os = connection.getOutputStream();
			IOHelper.copy(content, os);
			os.flush();
		} catch (IOException e) {
			throw new IOException("Could not write content to connection", e);
		} finally {
			IOHelper.closeQuietly(os);
		}
		
		int returnCode;
		try {
			returnCode = connection.getResponseCode();
		} catch (SocketTimeoutException e) {
			throw new ReadResponseTimeoutException("Could not get response code from connection ", e);
		} catch (IOException e) {
			throw new IOException("Could not get response code from connection", e);
		}
		return returnCode;
	}
	
	
	public static InputStream readResponse(HttpURLConnection connection, boolean closeConnection) throws IOException, HttpClientException {
		return readResponse(connection, closeConnection, true);
	}
	
	public static InputStream readResponse(HttpURLConnection connection, boolean closeConnection, boolean onlyCode200IsOk) throws IOException, HttpClientException {
		if (connection == null)
			throw new IllegalArgumentException("Connection cannot be null");
		
		int responseCode = connection.getResponseCode();
		
		InputStream is = null;
		
		try {
			boolean statusOk;
			if (onlyCode200IsOk)
				statusOk = responseCode == HttpURLConnection.HTTP_OK;
			else
				statusOk = ((int)responseCode / 100) == 2;
			
			if (statusOk) {
				is = cacheInputStream(connection.getInputStream());
			} else {
				InputStream responseStream = connection.getErrorStream();
				if (responseStream != null)
					is = cacheInputStream(responseStream);
				throw new HttpClientException(responseCode, String.format("%d %s", responseCode, connection.getURL()), is);
			}
			
		} finally {
			if (closeConnection)
				disconnectQuietly(connection);
		}
		
		return is;
		
	}
	
	
	public static InputStream cacheInputStream(InputStream is) throws IOException {
		WriteReadOutputStream sos = IOHelper.newSmartOutputStream();
		try {
			IOHelper.copy(is, sos);
		} finally {
			IOHelper.closeQuietly(is);
			IOHelper.closeQuietly(sos);
		}
		return sos.getInputStream();
	}

	
	
	public static void disconnectQuietly(HttpURLConnection conn) {
		try {
			if (conn != null)
				conn.disconnect();
		} catch (Exception e) {
			log.error(String.format("NOT DISCONNECTING SO QUIETLY FROM %s: %s", conn, e.getMessage()), e);
		}
	}
	
}
