package tech.espublico.pades.server.services.timestamper;

import java.security.NoSuchAlgorithmException;

import javax.xml.crypto.dsig.DigestMethod;

import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DERObjectIdentifier;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;

import tech.espublico.pades.server.helper.CheckArgument;

/**
 * Supported Algorithms
 */
public enum DigestAlgorithm {

	SHA1("SHA-1", "1.3.14.3.2.26", DigestMethod.SHA1),
	SHA256("SHA-256", "2.16.840.1.101.3.4.2.1", DigestMethod.SHA256),
	SHA512("SHA-512", "2.16.840.1.101.3.4.2.3", DigestMethod.SHA512);

	private String name;
	private String oid;
	private String xmlId;

	private DigestAlgorithm(String name, String oid, String xmlId) {
		this.name = name;
		this.oid = oid;
		this.xmlId = xmlId;
	}

	/**
	 * Return the algorithm corresponding to the name
	 */
	public static DigestAlgorithm getByName(String algoName) throws NoSuchAlgorithmException {
		CheckArgument.notNull("algoName", algoName);
		switch (algoName) {
		case "SHA-1":
		case "SHA1":
			return SHA1;
		case "SHA-256":
		case "SHA256":
			return SHA256;
		case "SHA-512":
		case "SHA512":
			return SHA512;
		}
		throw new NoSuchAlgorithmException("unsupported algo: " + algoName);
	}

	public String getName() {
		return name;
	}

	public String getOid() {
		return oid;
	}

	public String getXmlId() {
		return xmlId;
	}

	/**
	 * Gets the ASN.1 algorithm identifier structure corresponding to this digest algorithm
	 */
	public AlgorithmIdentifier getAlgorithmIdentifier() {
		/*
		 * The recommendation (cf. RFC 3380 section 2.1) is to omit the parameter for SHA-1, but some implementations
		 * still expect a NULL there. Therefore we always include a NULL parameter even with SHA-1, despite the
		 * recommendation, because the RFC states that implementations SHOULD support it as well anyway
		 */
		return new AlgorithmIdentifier(new DERObjectIdentifier(this.getOid()), new DERNull());
	}

}
