package tech.espublico.pades.server.services.storage;

import java.io.IOException;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Date;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tech.espublico.pades.server.di.Service;
import tech.espublico.pades.server.di.ServiceLocator;

@Service
public class FilesCleanupService {

	private static final Logger log = LoggerFactory.getLogger(FilesCleanupService.class);

	public static FilesCleanupService instance() {
		return ServiceLocator.INSTANCE.getInstance(FilesCleanupService.class);
	}

	public void cleanPathRecursive(Path dirToClean, long maxAgedMillis) throws IOException {
		if (!Files.isDirectory(dirToClean))
			throw new IllegalArgumentException("dir to clean must be directory");

		if (maxAgedMillis < 0)
			throw new IllegalArgumentException("maxAgewdMillis must be greather than 0");

		Files.walkFileTree(dirToClean, new SimpleFileVisitor<Path>() {

			@Override
			public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
				log.trace("+ File temp to delete {}", file.toAbsolutePath().toString());

				long lastAccessFile = attrs.lastAccessTime().toMillis();
				deleteDeadItem(file, lastAccessFile);

				return FileVisitResult.CONTINUE;
			}

			private void deleteDeadItem(Path item, long lastAccessFile) throws IOException {

				long currentTime = new Date().getTime();

				String type = item.toFile().isDirectory() ? "directory" : "file";
				if (lastAccessFile + maxAgedMillis < currentTime) {
					Files.delete(item);
					log.trace("   Temp {} deteled!", type);
				} else {
					log.trace("   Temp {} ignored, last acces was: {}; now: {}, max aged millis: {}", type, lastAccessFile, currentTime, maxAgedMillis);
				}
			}

			@Override
			public FileVisitResult postVisitDirectory(Path dir, IOException e) throws IOException {
				if (e == null) {
					try {
						if (Files.isSameFile(dirToClean, dir)) {
							log.trace("Current directory is tmp, ignore directory {}", dir.toAbsolutePath().toString());
						} else {
							log.trace("+ Current directory to delete {}", dir.toAbsolutePath().toString());
							deleteDeadItem(dir, dir.toFile().lastModified());
						}
					} catch (DirectoryNotEmptyException ignore) {
						//In another iteration of the walker we will end up dealeting the empty directory
					}
					return FileVisitResult.CONTINUE;

				} else {
					throw e;
				}
			}
		});
	}

}
