package tech.espublico.pades.server.services.scheduler;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.function.Function;

import org.quartz.CronScheduleBuilder;
import org.quartz.CronTrigger;
import org.quartz.Job;
import org.quartz.JobBuilder;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SchedulerFactory;
import org.quartz.TriggerBuilder;
import org.quartz.TriggerKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tech.espublico.pades.server.services.ConfigService;
import tech.espublico.pades.server.services.PropertyValidatorException;
import tech.espublico.pades.server.di.Service;
import tech.espublico.pades.server.di.ServiceLocator;
import tech.espublico.pades.server.services.storage.FilesCleanupJob;

/**
 * Service to register and start the scheduled taskts.
 * The default task is the temporal files cleaning job {@link FilesCleanupJob}
 */
@Service
public class SchedulerService {
	private static final Logger log = LoggerFactory.getLogger(SchedulerService.class);

	private static final String PREFIX_PROPERTIES = "application.job.";

	public static SchedulerService instance() {
		return ServiceLocator.INSTANCE.getInstance(SchedulerService.class);
	}

	private final Scheduler scheduler;
	private final Set<String> enabledJobs = new HashSet<>();
	private final Map<String, TriggerKey> triggers = new HashMap<>();
	private boolean started = false;

	public SchedulerService() throws SchedulerException, PropertyValidatorException {
		System.setProperty("org.terracotta.quartz.skipUpdateCheck", "true");

		Properties schedulerProperties;
		schedulerProperties = ConfigService.instance().subProperties("scheduler.org.quartz.", "org.quartz.");

		schedulerProperties.putIfAbsent("org.quartz.scheduler.instanceName", "Signer Server Scheduler");
		schedulerProperties.putIfAbsent("org.quartz.threadPool.threadCount", "1");
		schedulerProperties.putIfAbsent("org.quartz.jobStore.class", "org.quartz.simpl.RAMJobStore");

		SchedulerFactory schedFact = new org.quartz.impl.StdSchedulerFactory(schedulerProperties);
		scheduler = schedFact.getScheduler();

		registerJob(FilesCleanupJob.KEY, "Cleanup Temporal", FilesCleanupJob.class, FilesCleanupJob::map);
	}

	/**
	 * Launch scheduler service qartz
	 */
	public synchronized void start() {
		try {
			if (started)
				throw new RuntimeException("Scheduler service allready started");

			scheduler.start();

			started = true;
		} catch (SchedulerException e) {
			throw new RuntimeException("Error starting applicatio scheduler service", e);
		}
	}

	/**
	 * Stops the scheduler service
	 */
	public synchronized void stop() {
		try {
			log.info("Stopping application scheduler service...");
			this.scheduler.shutdown();
			started = false;
		} catch (SchedulerException e) {
			throw new RuntimeException("Error stopping application scheduler service", e);
		}
	}

	/**
	 * Register job in the Qartz service
	 *
	 * @param key
	 * 		key for the job configuration from properties
	 * @param name
	 * 		Name of the job
	 * @param jobClass
	 * 		class to register for the job
	 * @param map
	 * 		function to map the properties read to the job data
	 *
	 * @throws PropertyValidatorException
	 * 		in case of error while reading configuration properties
	 * @throws SchedulerException
	 * 		in case of not being able to register the job
	 */
	public void registerJob(String key, String name, Class<? extends Job> jobClass, Function<Properties, JobDataMap> map)
			throws PropertyValidatorException, SchedulerException {
		String group = "Internal";

		String jobSubsetPropertyKey = PREFIX_PROPERTIES + key + ".";
		Properties properties = ConfigService.instance()//
				.subProperties(jobSubsetPropertyKey, "");

		String cronString = properties.getProperty("cron");

		CronScheduleBuilder cron = CronScheduleBuilder.cronSchedule(cronString);
		CronTrigger trigger = TriggerBuilder.newTrigger()//
				.withIdentity(name, group)//
				.startNow()//
				.withSchedule(cron)//
				.build();//
		triggers.put(key, TriggerKey.triggerKey(name, group));

		JobDetail jobDetail = JobBuilder.newJob(jobClass)//
				.withIdentity(name, group)//
				.usingJobData(map.apply(properties))//
				.storeDurably()//
				.build();//

		scheduler.scheduleJob(jobDetail, trigger);
		log.debug("Job {} initialized and scheduled", key);
		log.warn("Next fire time for {}: {}", key, trigger.getNextFireTime().toString());

		this.enabledJobs.add(key);
	}

	/**
	 * @return the scheduler
	 */
	public Scheduler getScheduler() {
		return scheduler;
	}

	/**
	 * @return enabled jobs
	 */
	public Set<String> getEnabledJobs() {
		return enabledJobs;
	}

	/**
	 * The map keys are the job keys that correspond with the configuration keys.
	 *
	 * @return the triggers asociated with the correspondig jobs.
	 */
	public Map<String, TriggerKey> getTriggers() {
		return triggers;
	}
}
