package tech.espublico.pades.server.services.pades;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tech.espublico.pades.server.di.Service;
import tech.espublico.pades.server.di.ServiceLocator;
import tech.espublico.pades.server.exceptions.OpenSignatureException;
import tech.espublico.pades.server.exceptions.OpenSignatureException.OpenSignatureExceptionReason;
import tech.espublico.pades.server.helper.FilesHelper;
import tech.espublico.pades.server.helper.IOHelper;
import tech.espublico.pades.server.models.PDFSignedDocumentModel;
import tech.espublico.pades.server.models.PDFSignerModel;
import tech.espublico.pades.server.services.ConfigService;
import tech.espublico.pades.server.services.PropertyValidatorException;
import tech.espublico.pades.server.signers.service.PDFSignatureModel;
import tech.espublico.pades.server.signers.service.PDFSignatureModel.BarcodeType;
import tech.espublico.pades.server.signers.service.digest.hash.PDFDigestModel;
import tech.espublico.pades.server.signers.service.digest.hash.PDFHash;
import tech.espublico.pades.server.signers.service.digest.hash.PDFHashService;
import tech.espublico.pades.server.signers.service.digest.second.PDFSecondHash;
import tech.espublico.pades.server.signers.service.digest.second.PDFSecondHashModel;
import tech.espublico.pades.server.signers.service.digest.second.PDFSecondHashService;
import tech.espublico.pades.server.signers.service.finalize.TimedHashMap;
import tech.espublico.pades.server.signers.service.finalize.TimedHashMap.TimeOutMode;
import tech.espublico.pades.server.signers.service.prepare.PDFPrepareForSignatureService;
import tech.espublico.pades.server.signers.service.sign.PDFSignModel;
import tech.espublico.pades.server.signers.service.sign.PDFSignService;

@Service
public class PadesService {

	private static final Logger log = LoggerFactory.getLogger(PadesService.class);

	public static PadesService instance() {
		return ServiceLocator.INSTANCE.getInstance(PadesService.class);
	}

	private final PDFPrepareForSignatureService pdfPrepareForSignatureService;
	private final PDFHashService pdfHashService;
	private final PDFSecondHashService pdfSecondHashService;
	private final PDFSignService pdfSignService;

	private final TimedHashMap<String, PadesSession> sessionMap;

	public PadesService() throws PropertyValidatorException {
		this.pdfPrepareForSignatureService = PDFPrepareForSignatureService.instance();
		this.pdfHashService = PDFHashService.instance();
		this.pdfSecondHashService = PDFSecondHashService.instance();
		this.pdfSignService = PDFSignService.instance();

		long sessionExpiration = ConfigService.instance().getLong("pades-session.expiration", 60000L * 60);
		this.sessionMap = new TimedHashMap<>("pades-sessions", sessionExpiration, TimeOutMode.Relative, this::cleanSession);
	}

	private void cleanSession(PadesSession padesSession) {
		try {
			IOHelper.closeQuietly(padesSession.getPdfHash());

			for (Path path : padesSession.getTempPaths()) {
				Files.deleteIfExists(path);
			}
		} catch (Exception e) {
			log.error("Error while cleaning session", e);
		}
	}

	private void cleanAndDeleteSession(String sessionId) throws OpenSignatureException {
		this.cleanSession(this.getSession(sessionId));
		this.sessionMap.remove(sessionId);
	}

	/**
	 * This is neede to be able to upload files to the session
	 *
	 * @param sessionId
	 * 		session id
	 *
	 * @throws OpenSignatureException
	 * 		in the case the session is already registered
	 */
	public void createSession(String sessionId) throws OpenSignatureException {
		if (this.sessionMap.containsKey(sessionId))
			throw new OpenSignatureException("", OpenSignatureExceptionReason.UNKNOWN);

		this.sessionMap.put(sessionId, new PadesSession(sessionId));
	}

	/**
	 * Obtains an open session throwing error if the session does not exists
	 *
	 * @param sessionId
	 * 		session id
	 *
	 * @return the session model associated with the session id
	 *
	 * @throws OpenSignatureException
	 * 		in the case the session is not registered
	 */
	public PadesSession getSession(String sessionId) throws OpenSignatureException {
		if (!this.sessionMap.containsKey(sessionId))
			throw new OpenSignatureException("", OpenSignatureExceptionReason.UNKNOWN);

		return this.sessionMap.get(sessionId);
	}

	/**
	 * Adds a temporal file to the current session, to be able to deleted the file once the session is closed
	 *
	 * @param sessionId
	 * 		session id
	 * @param path
	 * 		file to add to the session
	 *
	 * @throws OpenSignatureException
	 * 		in the case the session is not registered
	 */
	public void addTempFile(String sessionId, Path path) throws OpenSignatureException {
		PadesSession padesSession = this.getSession(sessionId);
		padesSession.getTempPaths().add(path);
	}

	public Path prepare(String sessionId, PDFSignerModel pdfSignerModel) throws OpenSignatureException {
		try {
			PDFSignatureModel pdfSignatureModel = new PDFSignatureModel.Builder()//
					.setUnsignedPath(pdfSignerModel.getFileIn().toPath())//

					.setBarcodeImage(FilesHelper.toPath(pdfSignerModel.getBarcodeImage()))//
					.setBarcodeType(//
							pdfSignerModel.getBarcodeType() == null ?//
									null : BarcodeType.valueOf(pdfSignerModel.getBarcodeType().name()))//

					.setDownloadText(pdfSignerModel.getDownloadText())//
					.setPageCountLabel(pdfSignerModel.getPageCountLabel())//

					.setMark(pdfSignerModel.getMark())//
					.setSubMarkLeft(pdfSignerModel.getSubMarkLeft())//
					.setSubMarkRight(pdfSignerModel.getSubMarkRight())//

					.setSigners(pdfSignerModel.getSigners())//
					.setSignatureCountLabel(pdfSignerModel.getSignatureCountLabel())//
					.setVisibleSignature(pdfSignerModel.isVisibleSignature())//

					.setTotalSignatures(pdfSignerModel.getTotalSignatures())//
					.createPDFSignatureModel();
			return this.pdfPrepareForSignatureService.prepare(pdfSignatureModel);
		} finally {
			this.cleanAndDeleteSession(sessionId);
		}
	}

	public byte[] digest(String sessionId, PDFSignerModel pdfSignerModel) throws OpenSignatureException {
		PDFDigestModel pdfDigestModel = new PDFDigestModel.Builder()//
				.setCerts(pdfSignerModel.getCerts())//
				.setReason(pdfSignerModel.getReason())//
				.setLocation(pdfSignerModel.getLocation())//
				.setSignDate(pdfSignerModel.getSignDate())//
				.setSignatureNumber(pdfSignerModel.getSignatureNumber())//
				.setContactInfo(pdfSignerModel.getContactInfo())//
				.setCertifyNoChangesAllowed(pdfSignerModel.isCertifyNoChangesAllowed())//
				.setUserName(pdfSignerModel.getUserName())//
				.setUserCharge(pdfSignerModel.getUserCharge())//
				.setVisibleSignature(pdfSignerModel.isVisibleSignature())//
				.setSignatureCountLabel(pdfSignerModel.getSignatureCountLabel())//
				.setTotalSignatures(pdfSignerModel.getTotalSignatures())//
				.setSignatureDate(pdfSignerModel.getSignatureDate())//
				.setSignatureImage(FilesHelper.toPath(pdfSignerModel.getSignatureImage()))//
				.createPDFDigestModel();

		PDFHash pdfHash = this.pdfHashService.getHash(pdfDigestModel, pdfSignerModel.getFileIn().toPath());

		PDFSecondHashModel pdfSecondHashModel = new PDFSecondHashModel.Builder()//
				.setByteMode(pdfSignerModel.getByteMode())//
				.setSigningDate(pdfSignerModel.getSignDate())//
				.setCerts(pdfSignerModel.getCerts())//
				.setSignerOCSPConfiguration(pdfSignerModel.getSignerOCSPConfiguration())//
				.createPDFSecondHashModel();

		PDFSecondHash pdfSecondHash = //
				this.pdfSecondHashService.getSecondHashToSign(pdfSecondHashModel, pdfHash.getContentHash());

		PadesSession padesSession = this.sessionMap.computeIfAbsent(sessionId, PadesSession::new);
		padesSession.setPdfHash(pdfHash);
		padesSession.setPdfSecondHash(pdfSecondHash);

		return pdfSecondHash.getSecondHash();
	}

	public PDFSignedDocumentModel sign(String sessionId, PDFSignerModel pdfSignerModel, byte[] signedBytes, boolean requestTimeStamp)
			throws OpenSignatureException {

		PadesSession padesSession = this.getSession(sessionId);

		try {
			PDFHash pdfHash = padesSession.getPdfHash();
			PDFSecondHash pdfSecondHash = padesSession.getPdfSecondHash();

			PDFSignModel pdfSignModel = new PDFSignModel.Builder().setHash(pdfSecondHash.getSecondHash())//
					.setByteMode(pdfSignerModel.getByteMode())//
					.setPreparedPdf(pdfSignerModel.getFileIn().toPath())//
					.setPDFHash(pdfHash)//
					.setCerts(pdfSignerModel.getCerts())//
					.setTimestamperData(pdfSignerModel.getTimestamperData())//
					.setBackupTimestamperData(pdfSignerModel.getBackupTimestamperData())//
					.createPDFSignModel();

			byte[] pkcs7EncodeBytes = this.pdfSignService.getRncodedPkcs7(pdfSignModel, signedBytes, pdfSecondHash.getSignedAttributes(), requestTimeStamp);

			return this.pdfSignService.sign(pdfSignModel, pkcs7EncodeBytes);
		} finally {
			this.cleanAndDeleteSession(sessionId);
		}

	}

}
