package tech.espublico.pades.server.services.ocsp;

import java.math.BigInteger;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Security;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.CertPathValidatorException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateParsingException;
import java.security.cert.TrustAnchor;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Vector;

import org.bouncycastle.asn1.DERObjectIdentifier;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.ocsp.OCSPObjectIdentifiers;
import org.bouncycastle.asn1.x509.X509Extension;
import org.bouncycastle.asn1.x509.X509Extensions;
import org.bouncycastle.ocsp.BasicOCSPResp;
import org.bouncycastle.ocsp.CertificateID;
import org.bouncycastle.ocsp.CertificateStatus;
import org.bouncycastle.ocsp.OCSPException;
import org.bouncycastle.ocsp.OCSPReqGenerator;
import org.bouncycastle.ocsp.OCSPResp;
import org.bouncycastle.ocsp.RevokedStatus;
import org.bouncycastle.ocsp.SingleResp;
import org.bouncycastle.ocsp.UnknownStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import sun.security.provider.certpath.AlgorithmChecker;
import tech.espublico.pades.server.helper.CheckArgument;
import tech.espublico.pades.server.helper.CollectionsHelper;
import tech.espublico.pades.server.models.CertificateOCSPConfiguration;
import tech.espublico.pades.server.services.ConfigService;
import tech.espublico.pades.server.services.PropertyValidatorException;
import tech.espublico.pades.server.services.ocsp.OCSPValidationException.OCSPValidationExceptionReason;
import tech.espublico.pades.server.signers.sign.ocsp.OCSPService;
import tech.espublico.pades.server.di.Service;

@Service
public class OCSPValidationService implements OCSPService {

	public static final Logger log = LoggerFactory.getLogger(OCSPValidationService.class);

	private static final String KP_OCSP_SIGNING_OID = "1.3.6.1.5.5.7.3.9";

	private final boolean newOcspCertificateVerification;

	public OCSPValidationService() throws PropertyValidatorException {

		this.newOcspCertificateVerification = ConfigService.instance().getBool("certificate-manager.new-ocsp-certificate-verification", true);

		Security.addProvider(new org.bouncycastle.jce.provider.BouncyCastleProvider());
	}

	@Override
	public OCSPResp validateOCSPResponse(X509Certificate x509Certificate, Date time, CertificateOCSPConfiguration signerOCSPConfiguration,
			X509Certificate issuerCertificate) throws OCSPValidationException {
		OCSPValidationModel validationModel = new OCSPValidationModel(x509Certificate);
		validationModel.validAtTime(time);
		validationModel.withOCSPServer(signerOCSPConfiguration);
		validationModel.withIssuerCertificate(issuerCertificate);

		this.validateCertificate(validationModel);
		this.validateOCSPResponse(validationModel);

		return validationModel.getValidationResponse();
	}

	public void validateCertificate(OCSPValidationModel validationModel) throws OCSPValidationException {
		CheckArgument.notNull("Validation Model", validationModel);

		try {
			generateRequest(validationModel);
		} catch (OCSPException e) {
			throw new OCSPValidationException(OCSPValidationExceptionReason.CANT_CREATE_REQUEST, "Cant create ocsp request", e);
		}

		OCSPHttpClient.queryOCSPServer(validationModel);

		if (validationModel.getValidationResponse() == null) {
			throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_HTTP_RESPONSE, "No valid response from ocsp server");
		}
	}

	public void validateOCSPResponse(OCSPValidationModel validationModel) throws OCSPValidationException {
		CheckArgument.notNull("OCSP validation model", validationModel);
		CheckArgument.notNull("OCSP client configuration", validationModel.getOcspConfiguration());
		CheckArgument.notNull("OCSP validation response", validationModel.getValidationResponse());

		CertificateOCSPConfiguration ocspConfiguration = validationModel.getOcspConfiguration();

		if (!ocspConfiguration.isValidateResponse())
			return;

		BasicOCSPResp ocspResponse;
		try {
			Object o = validationModel.getValidationResponse().getResponseObject();
			ocspResponse = (BasicOCSPResp) o;
		} catch (ClassCastException | OCSPException e) {
			throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE,
					String.format("Invalid ocsp response object from server %s", ocspConfiguration.getServerURL().toString()), e);
		}

		log.debug(ocspResponse.toString());

		SingleResp[] responses = ocspResponse.getResponses();

		for (SingleResp resp : responses) {
			if (resp.getCertStatus() == CertificateStatus.GOOD) {
				// OK
			} else {
				Object certStatus = resp.getCertStatus();
				if (certStatus.getClass().isAssignableFrom(UnknownStatus.class)) {
					throw new OCSPValidationException(OCSPValidationExceptionReason.UNKNOWN_CERTIFICATE,
							String.format("Unknown certificate from server %s", ocspConfiguration.getServerURL().toString()));

				} else if (certStatus.getClass().isAssignableFrom(RevokedStatus.class)) {
					throw new OCSPValidationException(OCSPValidationExceptionReason.REVOKED_CERTIFICATE,
							String.format("Revoked certificate from server %s", ocspConfiguration.getServerURL().toString()));
				}
				throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE,
						String.format("Invalid response from server %s: %s", ocspConfiguration.getServerURL().toString(), certStatus.toString()));
			}

			X509Certificate[] ocspCertificates;
			try {
				ocspCertificates = ocspResponse.getCerts("BC");
			} catch (NoSuchProviderException e) {
				throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE, "Bouncy Castle is not registered ¿?", e);
			} catch (OCSPException e) {
				throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE, "Could not obtain ocsp certificates from response", e);
			}

			X509Certificate ocspCertificate = ocspCertificates[0];

			if (newOcspCertificateVerification) {
				newVerifyOcspCertificate(ocspCertificates, ocspConfiguration.getResponderCertificateCasList(), ocspResponse);
			} else {
				verifyOcspCertificate(ocspCertificate, validationModel.getIssuerCertificate(), ocspConfiguration.getResponderCertificateList(),
						validationModel.getValidAt(), ocspResponse);

				try {
					newVerifyOcspCertificate(ocspCertificates, ocspConfiguration.getResponderCertificateCasList(), ocspResponse);
				} catch (Throwable e) {
					log.error("newVerifyOcspCertificateLog", e);
				}
			}

			try {
				if (!ocspResponse.verify(ocspCertificate.getPublicKey(), "BC")) {
					throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE, "OcspResponse signature validation is invalid");
				}
			} catch (NoSuchProviderException e) {
				throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE, "Bouncy Castle is not registered ¿?", e);
			} catch (OCSPException e) {
				throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE, "Could not obtain ocsp certificates from response", e);
			}
		}
	}

	@SuppressWarnings("restriction")
	private void verifyOcspCertificate(X509Certificate ocspCertificate, X509Certificate issuerCertificate, List<X509Certificate> responderCertificateList,
			Date validAt, BasicOCSPResp ocspResponse) throws OCSPValidationException {

		if (ocspCertificate != null) {
			// Check if the response is signed by the issuing CA
			if (ocspCertificate.equals(issuerCertificate)) {
				// cert is trusted, now verify the signed response
				verifySignature(ocspCertificate, ocspResponse);

				// Check if the response is signed by a trusted responder
			} else if (CollectionsHelper.isNotEmpty(responderCertificateList) && containsResponseOcspCert(ocspCertificate, responderCertificateList)) {
				// cert is trusted, now verify the signed response
				verifySignature(ocspCertificate, ocspResponse);

				// Check if any of the configured ocsp certificates is valid
			} else if (CollectionsHelper.isNotEmpty(responderCertificateList) //
					&& !isAnyResponseOcspCertValid(responderCertificateList, validAt)) {
				throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE, "No valid ocsp certificates found for the current date");

				// Check if the response is signed by an authorized responder
			} else if (ocspCertificate.getIssuerX500Principal().equals(issuerCertificate.getSubjectX500Principal())) {
				// Check for the OCSPSigning key purpose
				try {
					List<String> keyPurposes = ocspCertificate.getExtendedKeyUsage();
					if (keyPurposes == null || !keyPurposes.contains(KP_OCSP_SIGNING_OID)) {
						throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE,
								"Responder's certificate not valid for signing OCSP responses");
					}
				} catch (CertificateParsingException cpe) {
					// assume cert is not valid for signing
					throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE,
							"Responder's certificate not valid for signing OCSP responses", cpe);
				}

				// Check algorithm constraints specified in security property "jdk.certpath.disabledAlgorithms".
				try {
					AlgorithmChecker algChecker = new AlgorithmChecker(new TrustAnchor(issuerCertificate, null), null);
					algChecker.init(false);
					algChecker.check(ocspCertificate, Collections.<String>emptySet());
				} catch (CertPathValidatorException e) {
					throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE,
							"Responder's certificate not valid for signing OCSP responses", e);
				}

				// check the validity
				try {
					if (validAt == null) {
						ocspCertificate.checkValidity();
					} else {
						ocspCertificate.checkValidity(validAt);
					}
				} catch (CertificateException e) {
					throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE,
							"Responder's certificate not within the validity period", e);
				}

				// check signature
				verifySignature(ocspCertificate, ocspResponse);

			} else {
				throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE,
						"Responder's certificate is not authorized to sign OCSP responses");
			}

		} else {
			throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE, "Ocsp certificate not found");
		}

	}

	private void newVerifyOcspCertificate(X509Certificate[] ocspCertificates, List<X509Certificate> responderCertificateCasList, BasicOCSPResp ocspResponse)
			throws OCSPValidationException {

		final X509Certificate ocspCertificate = ocspCertificates[0];

		if (ocspCertificate == null)
			throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE, "Ocsp certificate not found");

		boolean existsVerifiedIssuer = existsVerifiedCertificateIssuer(ocspCertificate, responderCertificateCasList);

		// Si no tenemos directamente el issuer comprobamos con la cadena de certificados
		int i = 1;
		X509Certificate certificate = ocspCertificate;
		while (!existsVerifiedIssuer && i < ocspCertificates.length) {
			final X509Certificate nextCertificate = ocspCertificates[i];

			if (nextCertificate == null || !isVerifiedCertificateIssuer(certificate, nextCertificate)) {
				throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE, "Error verifying responder's certificate");
			}
			existsVerifiedIssuer = existsVerifiedCertificateIssuer(nextCertificate, responderCertificateCasList);

			certificate = nextCertificate;
			i++;
		}

		if (existsVerifiedIssuer) {
			// now verify the signed response
			verifySignature(ocspCertificate, ocspResponse);
		} else {
			log.error("ocsp issuer not found: {}", ocspCertificate.getIssuerX500Principal());
			throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE,
					"Responder's certificate is not authorized to sign OCSP responses");
		}
	}

	private boolean existsVerifiedCertificateIssuer(final X509Certificate certificate, final List<X509Certificate> responderCertificateCasList)
			throws OCSPValidationException {
		return responderCertificateCasList.stream() //
				.anyMatch(ca -> isVerifiedCertificateIssuer(certificate, ca));
	}

	private boolean isVerifiedCertificateIssuer(final X509Certificate certificate, final X509Certificate issuerToCheck) {
		if (!issuerToCheck.getSubjectX500Principal().equals(certificate.getIssuerX500Principal()))
			return false;

		try {
			certificate.verify(issuerToCheck.getPublicKey());
		} catch (InvalidKeyException | CertificateException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
			return false;
		}

		return true;
	}

	private void verifySignature(X509Certificate ocspCertificate, BasicOCSPResp ocspResponse) throws OCSPValidationException {
		// check the signature
		try {
			Signature respSignature = Signature.getInstance(ocspResponse.getSignatureAlgName());
			respSignature.initVerify(ocspCertificate.getPublicKey());
			respSignature.update(ocspResponse.getTBSResponseData());

			if (!respSignature.verify(ocspResponse.getSignature()))
				throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE, "Error verifying signature of OCSP SignerResponse");

		} catch (InvalidKeyException | NoSuchAlgorithmException | SignatureException | OCSPException e) {
			throw new OCSPValidationException(OCSPValidationExceptionReason.INVALID_OCSP_RESPONSE, "Error verifying signature of OCSP SignerResponse", e);
		}

	}

	private boolean containsResponseOcspCert(X509Certificate ocspCertificate, List<X509Certificate> responderCertificateList) {
		return responderCertificateList.stream().anyMatch(cert -> cert.equals(ocspCertificate));
	}

	public static boolean between(Date x, Date from, Date to) {
		return x.compareTo(from) >= 0 && x.compareTo(to) <= 0;
	}

	private boolean isAnyResponseOcspCertValid(List<X509Certificate> responderCertificateList, Date validAt) {
		return responderCertificateList.stream().anyMatch(cert -> between(validAt, cert.getNotBefore(), cert.getNotAfter()));
	}

	private void generateRequest(OCSPValidationModel model) throws OCSPException {

		X509Certificate certificadoX509 = model.getCertificate();
		X509Certificate certificadoX509Emisor = model.getIssuerCertificate();
		Date validTime = model.getValidAt();

		// 1 - id generation
		CertificateID id = new CertificateID(CertificateID.HASH_SHA1, certificadoX509Emisor, certificadoX509.getSerialNumber());

		// 2- ocps request generation
		OCSPReqGenerator requestGenerator = new OCSPReqGenerator();
		requestGenerator.addRequest(id);

		// 3- Query extensions needed RFC 2560
		BigInteger time = BigInteger.valueOf(validTime.getTime()); // Signature time
		Vector<DERObjectIdentifier> oids = new Vector<DERObjectIdentifier>();
		oids.add(OCSPObjectIdentifiers.id_pkix_ocsp_nonce);
		Vector<X509Extension> values = new Vector<X509Extension>();
		values.add(new X509Extension(false, new DEROctetString(time.toByteArray())));

		// 4. Add query extesions
		requestGenerator.setRequestExtensions(new X509Extensions(oids, values));

		// OCSP Query generation
		model.setValidationRequest(requestGenerator.generate());
	}

}
