package tech.espublico.pades.server.services.ocsp;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.bouncycastle.asn1.ocsp.OCSPResponseStatus;
import org.bouncycastle.ocsp.OCSPReq;
import org.bouncycastle.ocsp.OCSPResp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tech.espublico.pades.server.helper.CheckArgument;
import tech.espublico.pades.server.helper.IOHelper;
import tech.espublico.pades.server.helper.WriteReadOutputStream;
import tech.espublico.pades.server.models.CertificateOCSPConfiguration;

/**
 * Implementation of certificate validation OCSP compatible with RFC 2560. (Online Certificate Status Protocol)
 */
public abstract class OCSPHttpClient {
	
	private static Logger log = LoggerFactory.getLogger(OCSPHttpClient.class);
	
	private static final String REQUEST_CONTENT_TYPE = "application/ocsp-request";
	private static final String RESPONSE_CONTENT_TYPE = "application/ocsp-response";
	private static final int maxRetryCount = 5;

	public static void queryOCSPServer(OCSPValidationModel validationModel)
			throws OCSPHttpClientException {
		
		CheckArgument.notNull("model", validationModel);
		CheckArgument.notNull("request", validationModel.getValidationRequest());
		
		OCSPResp ocspResponse = null;
		int count = 0;
		do {
			try {
				try {
					ocspResponse = postHttpOCSPQuery(validationModel);
				} catch (java.net.UnknownHostException e) {
					throw new OCSPHttpClientException(validationModel.getOcspConfiguration(), "UNKNOWN_HOST", false, e);
					
				} catch (java.net.ConnectException e) {
					throw new OCSPHttpClientException(validationModel.getOcspConfiguration(), "CONNECT_EXCEPTION", true, e);
					
				} catch (IOException e) {
					throw new OCSPHttpClientException(validationModel.getOcspConfiguration(), "IO_EXCEPTION", true, e);
				}
	
				switch(ocspResponse.getStatus()) {
				case OCSPResponseStatus.SUCCESSFUL:
					break;
					
				case OCSPResponseStatus.MALFORMED_REQUEST:
					throw new OCSPHttpClientException(validationModel.getOcspConfiguration(), "MALFORMED_REQUEST", false);
					
				case OCSPResponseStatus.SIG_REQUIRED:
					throw new OCSPHttpClientException(validationModel.getOcspConfiguration(), "SIG_REQUIRED", false);
					
				case OCSPResponseStatus.INTERNAL_ERROR:
					throw new OCSPHttpClientException(validationModel.getOcspConfiguration(), "INTERNAL_ERROR", false);
					
				case OCSPResponseStatus.UNAUTHORIZED:
					throw new OCSPHttpClientException(validationModel.getOcspConfiguration(), "UNAUTHORIZED", false);
					
				case OCSPResponseStatus.TRY_LATER:
					throw new OCSPHttpClientException(validationModel.getOcspConfiguration(), "TRY_LATER", true);
					
				default:
					throw new OCSPHttpClientException(validationModel.getOcspConfiguration(), "UNKNOWN", true);

				}
				
			} catch (OCSPHttpClientException e) {
				ocspResponse = null;
				log.warn(String.format("OCSPHttpClientException %s", e.toString()), e);
				if (e.isCanRetry()) {
					count++;
					log.debug(String.format("Retrying OSCP HTTP Query %d after sleeping a while", count));
					if (count < maxRetryCount) {
						try {
							Thread.sleep(200 * count);
						} catch (InterruptedException e1) {
							log.info("Interrumpted OCSP http client thread wait, exiting");
							count = maxRetryCount;
						}
						log.info(
								String.format("Retrying OSCP HTTP Query %d of %d to server %s", count, maxRetryCount, validationModel.getOcspConfiguration().toString()));
					} else {
						log.info(String.format("OSCP HTTP %s client has no more query tries left, exiting", validationModel.getOcspConfiguration().toString()));
						throw e;
					}
				} else {
					throw e;
				}
			}
			
		} while (count < maxRetryCount && ocspResponse == null);
		
		validationModel.setValidationResponse(ocspResponse);
	}
	

	private static OCSPResp postHttpOCSPQuery(OCSPValidationModel validationModel)
			throws IOException, OCSPHttpClientException {
		
		CertificateOCSPConfiguration ocspServerConfiguration = validationModel.getOcspConfiguration();
		CheckArgument.notNull("OCSP Server Configuration", ocspServerConfiguration);
		if (!ocspServerConfiguration.isCheckOCSP())
			return null;
		
		OCSPReq request = validationModel.getValidationRequest();
		CheckArgument.notNull("OCSPRequest", request);
		
		// OCSP request bytes
		byte[] byteRequest;
		try {
			byteRequest = request.getEncoded();
		} catch (IOException e) {
			throw new OCSPHttpClientException(ocspServerConfiguration, "Could not encode OCSP SignerRequest as a byte array", false, e);
		}
		
		ByteArrayInputStream bin = new ByteArrayInputStream(byteRequest);
		HttpURLConnection con = null;

		try {
			con = (HttpURLConnection) ocspServerConfiguration.getServerURL().openConnection();
			
			con.setRequestProperty("Content-Type", REQUEST_CONTENT_TYPE);
			con.setRequestProperty("Accept", RESPONSE_CONTENT_TYPE);
			con.setFixedLengthStreamingMode(byteRequest.length);
			con.setDoOutput(true);
			
			OutputStream out = null;
			try {
				out = con.getOutputStream();
				IOHelper.copy(bin, out);
				out.flush();
			} finally {
				IOHelper.closeQuietly(out);
				IOHelper.closeQuietly(bin);
			}
			
			switch (con.getResponseCode()) {
			case HttpURLConnection.HTTP_OK:
				// OK
				break;
				
			default:
				throw new OCSPHttpClientException(ocspServerConfiguration,
						String.format("Ocsp server returned error code %d: %s", con.getResponseCode(), con.getResponseMessage()),
						true);
			}
			
			WriteReadOutputStream os;
			if (con.getContentLength() != -1) {
				os = IOHelper.newSmartOutputStream(con.getContentLength());
			} else {
				os = IOHelper.newSmartOutputStream();
			}
			
			InputStream in = null;
			try {
				in = (InputStream) con.getContent();
				IOHelper.copy(in, os);
			} finally {
				IOHelper.closeQuietly(in);
				IOHelper.closeQuietly(os);
				in = null;
			}
			
			try {
				in = os.getInputStream();
				OCSPResp response = new OCSPResp(in);
				
				return response;
			} catch (IOException e){
				try (InputStream input = os.getInputStream()) {
					String base64Input = Base64.encodeBase64String(IOUtils.toByteArray(input));
					log.error("Error in OCSPResp. The input in base64 is = {}", base64Input, e);
				} finally {
					throw(e);
				}			       				
			} finally {
				IOHelper.closeQuietly(in);
			}
			
		} finally {
			if (con != null)
				con.disconnect();
			
		}
	}

}
