package tech.espublico.pades.server.services;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.lang.reflect.Type;
import java.nio.file.Path;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Objects;

import com.google.gson.FieldNamingPolicy;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.LongSerializationPolicy;
import com.google.gson.stream.JsonWriter;

import tech.espublico.pades.server.services.gson.ByteArrayTypeAdapter;
import tech.espublico.pades.server.services.gson.CertificateTypeAdapter;
import tech.espublico.pades.server.services.gson.FileTypeAdapter;
import tech.espublico.pades.server.services.gson.X509CertificateTypeAdapter;
import tech.espublico.pades.server.di.Service;
import tech.espublico.pades.server.di.ServiceLocator;
import tech.espublico.pades.server.services.gson.PathTypeAdapter;

/**
 * Service to transform to json objects and serilize to strings.
 */
@Service
public class GsonService {

	private final Gson gson;

	public static GsonService instance() {
		return ServiceLocator.INSTANCE.getInstance(GsonService.class);
	}

	public GsonService() throws CertificateException {
		GsonBuilder builder = new GsonBuilder();
		builder.setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES);
		builder.setLongSerializationPolicy(LongSerializationPolicy.STRING);
		builder.disableHtmlEscaping();

		builder.registerTypeAdapter(byte[].class, new ByteArrayTypeAdapter());
		builder.registerTypeAdapter(Certificate.class, new CertificateTypeAdapter());
		builder.registerTypeAdapter(X509Certificate.class, new X509CertificateTypeAdapter());
		builder.registerTypeHierarchyAdapter(Path.class, new PathTypeAdapter());
		builder.registerTypeHierarchyAdapter(File.class, new FileTypeAdapter());


		builder.setPrettyPrinting();
		gson = builder.create();
	}

	public Gson getGson() {
		return this.gson;
	}

	public <T> T fromJson(InputStream inputStream, Type type) throws IOException {
		try (InputStreamReader inputStreamReader = new InputStreamReader(inputStream)) {
			return this.gson.fromJson(inputStreamReader, type);
		}
	}
	public <T> T fromJson(InputStream inputStream, Class<T> clazz) throws IOException {
		try (InputStreamReader inputStreamReader = new InputStreamReader(inputStream)) {
			return this.gson.fromJson(inputStreamReader, clazz);
		}
	}

	public <T> void toJson(T object, Class<T> clazz, OutputStream outputStream) throws IOException {
		Objects.requireNonNull(object, "Expected object");

		try (OutputStreamWriter outputStreamWriter = new OutputStreamWriter(outputStream);
				JsonWriter jsonWriter = new JsonWriter(outputStreamWriter)) {
			gson.toJson(object, clazz, jsonWriter);
		}
	}

	public <T> String toJson(T object) {
		if (object == null)
			return null;

		return gson.toJson(object).trim();
	}

	public <T> T fromJson(String json, Type type) {
		return gson.fromJson(json, type);
	}

	public <T> T fromJson(String json, Class<T> clazz) {
		if (json == null || "".equals(json.trim()))
			return null;

		return (T) gson.fromJson(json, clazz);
	}
}
