package tech.espublico.pades.server.services;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Properties;

public class ExtProperties extends Properties {

	private Path basePath;

	void setBasePath(Path basePath) {
		this.basePath = basePath;
	}

	private String get(String key, String defaultValue) throws PropertyValidatorException {
		if (super.containsKey(key)) {
			return super.getProperty(key);
		}
		return defaultValue;
	}

	private String get(String key) throws PropertyValidatorException {
		if (super.containsKey(key)) {
			return super.getProperty(key).trim();
		}
		throw new PropertyValidatorException(key);
	}

	public Path getPath(String key, boolean shouldExist) throws PropertyValidatorException {
		Path p = Paths.get(this.basePath.toString(), getString(key));
		if (shouldExist && Files.notExists(p))
			throw new PropertyValidatorException(String.format("Path [%s] in key [%s] should exist", p, key));
		return p;
	}

	public Path getPath(String key) throws PropertyValidatorException {
		return this.getPath(key, true);
	}

	public String[] getStringArray(String key) throws PropertyValidatorException {
		String value = this.getString(key);
		return trim(value.split(","));
	}

	public String[] getStringArray(String key, String[] defVal) throws PropertyValidatorException {
		String value = this.get(key, null);
		if (value == null)
			return defVal;
		return trim(value.split(","));
	}

	public String[] trim(String[] arr) {
		for (int i = 0; i < arr.length; i++) {
			arr[i] = arr[i].trim();
		}
		return arr;
	}

	public int getInt(String key, int defVal) throws PropertyValidatorException {
		return Integer.valueOf(this.get(key, String.valueOf(defVal)));
	}

	public int getInt(String key) throws PropertyValidatorException {
		try {
			return Integer.valueOf(this.get(key));
		} catch (NumberFormatException e) {
			throw new PropertyValidatorException(key, e);
		}
	}

	public long getLong(String key) throws PropertyValidatorException {
		try {
			return Long.valueOf(this.get(key));
		} catch (NumberFormatException e) {
			throw new PropertyValidatorException(key, e);
		}
	}

	public long getLong(String key, long defaultVal) throws PropertyValidatorException {
		try {
			return Long.valueOf(this.get(key, String.valueOf(defaultVal)));
		} catch (NumberFormatException e) {
			throw new PropertyValidatorException(key, e);
		}
	}

	public boolean getBool(String key) throws PropertyValidatorException {
		return Boolean.valueOf(this.get(key));
	}

	public boolean getBool(String key, boolean defaultVal) throws PropertyValidatorException {
		return Boolean.valueOf(this.get(key, String.valueOf(defaultVal)));
	}

	public String getString(String key) throws PropertyValidatorException {
		return this.get(key);
	}

	public Properties subProperties(String prefix, String replacement) {
		Properties subProperties = new Properties();
		int replaceLength = prefix.length();
		for (String key : super.stringPropertyNames()) {
			if (key.startsWith(prefix)) {
				String realkey;
				if (replacement == null) {
					realkey = key;
				} else {
					realkey = key.substring(replaceLength);
				}

				subProperties.setProperty(realkey, super.getProperty(key));
			}
		}

		return subProperties;
	}

}
