package tech.espublico.pades.server.services;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.AbstractCollection;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.joran.JoranConfigurator;
import ch.qos.logback.core.joran.spi.JoranException;
import tech.espublico.pades.server.di.Service;
import tech.espublico.pades.server.di.ServiceLocator;

@Service
public class ConfigService {

	private static final Logger log = LoggerFactory.getLogger(ConfigService.class);

	private static String directConfigURL;

	public static ConfigService instance() {
		return ServiceLocator.INSTANCE.getInstance(ConfigService.class);
	}

	private final ExtProperties properties = new ExtProperties();
	private Path basePath;

	public ConfigService() {
		try {
			loadConfig(new File(ConfigService.directConfigURL));
		} catch (Exception e) {
			log.error("Could not initialize ConfigurationService", e);
			throw new RuntimeException(e);
		}
	}

	public static void startLogBack(String logbackFilePath) throws JoranException {
		if (logbackFilePath == null)
			return;

		LoggerContext context = (LoggerContext) LoggerFactory.getILoggerFactory();
		JoranConfigurator configurator = new JoranConfigurator();
		configurator.setContext(context);
		configurator.doConfigure(Paths.get(logbackFilePath).toFile());
	}

	public static void setDirectConfigURL(String directConfigURL) {
		ConfigService.directConfigURL = directConfigURL;
	}

	private void loadConfig(File file) throws PropertyValidatorException, IOException {

		Properties properties = new Properties();
		FileReader fileReader = null;
		try {
			fileReader = new FileReader(file);
			properties.load(fileReader);
		} catch (IOException e) {
			throw new PropertyValidatorException("Cannot read properties", file);
		} finally {
			if (fileReader != null) {
				fileReader.close();
			}
		}
		this.properties.setBasePath(file.toPath().getParent());
		this.properties.putAll(properties);
	}

	public Path getPath(String key) throws PropertyValidatorException {
		return this.properties.getPath(key);
	}

	public Path getPath(String key, boolean shouldExist) throws PropertyValidatorException {
		return this.properties.getPath(key, shouldExist);
	}

	public int getInt(String key) throws PropertyValidatorException {
		return this.properties.getInt(key);
	}

	public long getLong(String key) throws PropertyValidatorException {
		return this.properties.getLong(key);
	}

	public long getLong(String key, long defaultVal) throws PropertyValidatorException {
		return this.properties.getLong(key, defaultVal);
	}

	public boolean getBool(String key) throws PropertyValidatorException {
		return this.properties.getBool(key);
	}

	public boolean getBool(String key, boolean defaultVal) throws PropertyValidatorException {
		return this.properties.getBool(key, defaultVal);
	}

	public String getString(String key) throws PropertyValidatorException {
		return this.properties.getString(key);
	}

	public String[] getStringArray(String key, String[] defVal) throws PropertyValidatorException {
		return this.properties.getStringArray(key, defVal);
	}
		public String[] getStringArray(String key) throws PropertyValidatorException {
		return this.properties.getStringArray(key);
	}

	public ExtProperties subProperties(String prefix, String replacement) {
		ExtProperties subProperties = new ExtProperties();
		int replaceLength = prefix.length();
		for (String key : this.properties.stringPropertyNames()) {
			if (key.startsWith(prefix)) {
				String realkey;
				if (replacement == null) {
					realkey = key;
				} else {
					realkey = key.substring(replaceLength);
				}

				subProperties.setProperty(realkey, this.properties.getProperty(key));
			}
		}

		return subProperties;
	}
}
